Ext.namespace('Zarafa.plugins.placecall.dialogs');

/**
 * @class Zarafa.plugins.placecall.dialogs.PrepareCallDialog
 * @extends Zarafa.core.ui.Dialog
 *
 * Dialog to let the user choose the actual call parameters.
 */
Zarafa.plugins.placecall.dialogs.PrepareCallDialog = Ext.extend(Zarafa.core.ui.Dialog, {

	/**
	 * @constructor
	 * @param config Configuration structure
	 */
	constructor : function(config)
	{
		config = config || {};

		var params = config.params;
		var callerReadOnly = true;
		var caller = '';
		var callees = [];
		
		if(!Ext.isEmpty(params)) {
			callerReadOnly = params.callerReadOnly;
			caller = params.caller;
			callees = params.callees;
		}
		
		var calleeRadios = [];
		
		for(var key in callees) {
			calleeRadios.push({
				xtype: 'radio',
	        	boxLabel: _(key, 'plugin_placecall') + ': ' + callees[key],
	        	name: "callee",
        		inputValue : callees[key]
			});
		}

		Ext.applyIf(config, {
			title  : _('Call contact', 'plugin_placecall'),
			layout : 'fit',
			width  : 400,
			height : 120,
			items  : [{
				xtype  : 'form',
				layout : 'form',
				border : false,
				ref    : 'form',
				items  : [{
						xtype		: 'textfield',
						name		: 'caller',
						value		: caller,
						allowBlank	: false,
						readOnly	: callerReadOnly,
						fieldLabel	: _('Call via', 'plugin_placecall'),
						anchor		: '100%'
					}, {
						xtype : 'radiogroup',
						name : 'callee',
					    columns: 1,
						fieldLabel : _('Call number', 'plugin_placecall'),
						value : calleeRadios[0].inputValue,
					    items : calleeRadios
					}
				],
				buttons: [{
						text: _('Ok'),
						handler: this.onOk,
						scope: this
					},
					{
						text: _('Cancel'),
						handler: this.onCancel,
						scope: this
					}
				]
			}]
		});

		//Call superclass constructor
		Zarafa.plugins.placecall.dialogs.PrepareCallDialog.superclass.constructor.call(this, config);
	},
	
	onOk : function()
	{
		var form = this.form.getForm();
		
		if(form.isValid()) {
			var values = form.getValues();
			
			container.getRequest().singleRequest('placecallmodule', 'placecall', {
				caller : values['caller'],
				callee : values['callee'],
				profile : 'placecall'
			});
			this.close();
		}
	},
	
	onCancel : function()
	{
		this.close();
	}

});

//Register the dialog xtype
Zarafa.core.ui.Dialog.register(Zarafa.plugins.placecall.dialogs.PrepareCallDialog, 'placecall.preparecalldialog');
Ext.namespace('Zarafa.plugins.placecall.data');

/**
 * @class Zarafa.plugins.placecall.data.PrepareCallRecordResponseHandler
 * @extends Zarafa.core.data.AbstractResponseHandler
 *
 * Response handler dealing with prepare call data returned by the backend in order to choose the call details.
 */
Zarafa.plugins.placecall.data.PrepareCallRecordResponseHandler = Ext.extend(Zarafa.core.data.AbstractResponseHandler, {

	/**
	 * Evaluates the returned data object and either shows a message box in case the information is not
	 * sufficient or invokes the PrepareCallDialog to let the user initiate the call.
	 * @param {Object} response Object contained the response data.
	 */
	doItem : function(response) {
		var message = response.message;
		
		if(Ext.isEmpty(message)) {
			var params = {
					callerReadOnly	: response.callerReadOnly,
					caller			: response.caller,
					callees			: response.callees
				};
			
			Zarafa.plugins.placecall.dialogs.PrepareCallDialog.create({
				params : params
			});
		} else {
			Ext.MessageBox.show({
				title   : _('Warning'),
				msg     : message,
				icon    : Ext.MessageBox.WARNING,
				buttons : Ext.MessageBox.OK
			});
		}
	}
});

Ext.reg('placecall.preparecallrecordresponsehandler', Zarafa.plugins.placecall.data.PrepareCallRecordResponseHandler);
Ext.namespace('Zarafa.plugins.placecall');

/**
 * @class Zarafa.plugins.placecall.PlacecallPlugin
 * @extends Zarafa.core.Plugin
 *
 * This class integrates the plugin into the WebApp and allows
 * the user to place calls to a contact's phone numbers.
 */
Zarafa.plugins.placecall.PlacecallPlugin = Ext.extend(Zarafa.core.Plugin, {

	/**
	 * @constructor
	 * @param {Object} config Configuration object
	 *
	 */
	constructor : function (config)
	{
		config = config || {};
		Ext.applyIf(config, {
			name : 'placecall'
		});
		Zarafa.plugins.placecall.PlacecallPlugin.superclass.constructor.call(this, config);
		this.init();
	},

	/**
	 * Called after constructor.
	 * Registers insertion points in the contact context menu for place call invokation.
	 */
	init : function()
	{
		this.registerInsertionPoint('context.contact.contextmenu.options', this.putPlaceCallFromContactInContextMenu, this)
	},

	/**
	 * Creates the place call command entry for Contact objects.
	 *
	 * @return {Zarafa.core.ui.menu.ConditionalItem} menuItem MenuItem instance
	 */
	putPlaceCallFromContactInContextMenu : function()
	{
		var menuitem = {
			xtype				: 'zarafa.conditionalitem',
			text				: _('Call contact', 'plugin_placecall'),
			overflowText		: _('Call contact', 'plugin_placecall'),
			iconCls				: 'icon_placecall',
			singleSelectOnly	: true,
			handler				: this.placeCallFromContact,
			scope				: this
		};
		return menuitem;
	},

	/**
	 * Invoke the place call dialog for the currently selected contact.
	 *
	 * @param {Object} menuitem The menu item which was clicked
	 */
	placeCallFromContact : function(menuitem)
	{
		var selectedRecords = menuitem.getRecords();
		var selectedRecord = selectedRecords[0];
		
		container.getRequest().singleRequest('placecallmodule', 'preparefromcontact', {
			entryid : selectedRecord.get('entryid'),
			parent_entryid : selectedRecord.get('parent_entryid'),
			store_entryid : selectedRecord.get('store_entryid'),
			profile : 'placecall'
		}, new Zarafa.plugins.placecall.data.PrepareCallRecordResponseHandler());
	}

});

Zarafa.onReady(function() {
	if(container.getSettingsModel().get('zarafa/v1/plugins/placecall/enable') === true) {
		container.registerPlugin(new Zarafa.plugins.placecall.PlacecallPlugin());
	}
});
