<?php
/*
 * Copyright 2012 Holger de Carne
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License, version 3, 
 * as published by the Free Software Foundation with the following additional 
 * term according to sec. 7:
 *  
 * According to sec. 7 of the GNU Affero General Public License, version
 * 3, the terms of the AGPL are supplemented with the following terms:
 * 
 * "Zarafa" is a registered trademark of Zarafa B.V. The licensing of
 * the Program under the AGPL does not imply a trademark license.
 * Therefore any rights, title and interest in our trademarks remain
 * entirely with us.
 * 
 * However, if you propagate an unmodified version of the Program you are
 * allowed to use the term "Zarafa" to indicate that you distribute the
 * Program. Furthermore you may use our trademarks where it is necessary
 * to indicate the intended purpose of a product or service provided you
 * use it in accordance with honest practices in industrial or commercial
 * matters.  If you want to propagate modified versions of the Program
 * under the name "Zarafa" or "Zarafa Server", you may only do so if you
 * have a written permission by Zarafa B.V. (to acquire a permission
 * please contact Zarafa at trademark@zarafa.com).
 * 
 * The interactive user interface of the software displays an attribution
 * notice containing the term "Zarafa" and/or the logo of Zarafa.
 * Interactive user interfaces of unmodified and modified versions must
 * display Appropriate Legal Notices according to sec. 5 of the GNU
 * Affero General Public License, version 3, when you propagate
 * unmodified or modified versions of the Program. In accordance with
 * sec. 7 b) of the GNU Affero General Public License, version 3, these
 * Appropriate Legal Notices must retain the logo of Zarafa or display
 * the words "Initial Development by Zarafa" if the display of the logo
 * is not reasonably feasible for technical reasons. The use of the logo
 * of Zarafa in Legal Notices is allowed for unmodified and modified
 * versions of the software.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero General Public License for more details.
 *  
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

/**
 * Placecall Plugin module
 *
 * This class processes the call invokations from the client and
 * triggers the PBX system accordingly.
 */
class PlacecallModule extends Module {

	/* Phone properties for callee selection */
	private static $phonePropKeys = array(
				'business'  => PR_BUSINESS_TELEPHONE_NUMBER,
				'business2' => PR_BUSINESS2_TELEPHONE_NUMBER,
				'home'      => PR_HOME_TELEPHONE_NUMBER,
				'home2'     => PR_HOME2_TELEPHONE_NUMBER,
				'primary'   => PR_PRIMARY_TELEPHONE_NUMBER,
				'callback'  => PR_CALLBACK_TELEPHONE_NUMBER,
				'mobile'    => PR_MOBILE_TELEPHONE_NUMBER,
				'radio'     => PR_RADIO_TELEPHONE_NUMBER,
				'car'       => PR_CAR_TELEPHONE_NUMBER,
				'other'     => PR_OTHER_TELEPHONE_NUMBER,
				'pager'     => PR_PAGER_TELEPHONE_NUMBER,
				'assistant' => PR_ASSISTANT_TELEPHONE_NUMBER
			);

	/**
	 * Constructor
	 * @param int $id unique id.
	 * @param array $data list of all actions.
	 */
	public function PlacecallModule($id, $data) {
		parent::Module($id, $data);
	}

	/**
	 * Process the incoming events that were fired by the client.
	 * @return boolean True if everything was processed correctly.
	 */
	public function execute() {
		$result = false;
		foreach($this->data as $actionType => $actionData) {
			if(isset($actionType)) {
				$this->log("DEBUG: enter::{$actionType}");
				try {
					switch($actionType) {
						case 'preparefromcontact':
							$result = $this->prepareFromContact($actionData);
							break;
						case 'placecall':
							$result = $this->placeCall($actionData);
							break;
						default:
							$this->handleUnknownActionType($actionType);
					}
				} catch (MAPIException $e) {
					$this->log("ERROR: Exception '{$e}'");
					$this->sendFeedback(false, $this->errorDetailsFromException($e));
				}
				$resultString = ($result ? 'true' : 'false');
				$this->log("DEBUG: exit::{$actionType} (result={$resultString})");
			}
		}
		return $result;
	}
	
	/**
	 * Prepare call based upon a contact.
	 * @param mixed $action The data that holds the action request data.
	 * @return boolean True if the action succeeded.
	 **/
	private function prepareFromContact($action) {
		$store = $this->getActionStore($action);
		$store || $this->log('ERROR: Unable to open store'); 
		$entryid = $this->getActionEntryID($action);
		$entryid || $this->log('ERROR: Unable to determine entry id');
		$contact = mapi_msgstore_openentry($store, $entryid);
		$contact || $this->log('ERROR: Unable to get contact entry');
		return $this->prepareResponse($contact);
	}

	private function prepareResponse($contact) {
		$result = false;
		$response = array();
		$caller = $this->prepareCaller();
		if($caller !== false) {
			$callees = $this->prepareCallees($contact);
			if($callees !== false) {
				if(count($callees) > 0) {
					$response['callerReadOnly'] = PLUGIN_PLACECALL_CALLER_PROPERTY !== false;
					$response['caller'] = $caller;
					$response['callees'] = $callees;
					$result = true;
				} else {
					$response['message'] = 'Contact has no telephone numbers assigned.';
				}
			} else {
				$response['message'] = 'Unable to retrieve the contact information.';
			}
		} else {
			$response['message'] = 'Unable to determine your caller id. Please contact your administrator.';
		}
		$this->addActionData("item", $response);
		$bus = $this->getBus();
		$bus->addData($this->getResponseData());
		return $result;
	}
	
	private function prepareCaller() {
		$caller = false;
		if(PLUGIN_PLACECALL_CALLER_PROPERTY) {
			$caller = $this->getUserCaller();			
		} else {
			$settings = $this->getSettings();
			$caller = $settings->get('zarafa/v1/plugins/placecall/caller', '');
		}
		return $caller;
	}
	
	private function prepareCallees($contact) {
		$callees = false;
		$contactProps = mapi_getprops($contact, self::$phonePropKeys);
		$contactProps || $this->log('ERROR: Unable to retrieve contact properties');
		if($contactProps) {
			$callees = array();
			$this->log('DEBUG: Collecting callees for contact:');
			foreach(self::$phonePropKeys as $phonePropKeyId => $phonePropKey) {
				$callee = MAPI_E_NOT_FOUND;
				if(isset($contactProps[$phonePropKey])) {
					$callee = $contactProps[$phonePropKey];
				}
				if($callee !== MAPI_E_NOT_FOUND && ($callee = trim($callee)) != '') {
					$callees[$phonePropKeyId] = $callee;
					$this->log("DEBUG: Collecting phone number '{$phonePropKeyId}' = '{$callee}'");
				} else {
					$this->log("DEBUG: Skipping undefined phone number '{$phonePropKeyId}'");
				}
			}
		}
		return $callees;
	}
	
	/**
	 * Place a call for the selected numbers.
	 * @param mixed $action The data that holds the action request data.
	 * @return boolean True if the action succeeded.
	 **/
	private function placeCall($action) {
		$result = false;
		$caller = $this->resolveActionCaller($action);
		$this->log("DEBUG: caller = '{$caller}'");
		if($caller) {
			$callee = $this->resolveActionCallee($action);
			$this->log("DEBUG: callee = '{$callee}'");
			if($callee) {
				switch(PLUGIN_PLACECALL_DRIVER) {
					case 'astspool':
						$result = $this->astspoolPlaceCall($caller, $callee);
						break;
					case 'astmgr':
						$result = $this->astmgrPlaceCall($caller, $callee);
						break;
					default:
						$this->log("ERROR: Unknown driver option '".PLUGIN_PLACECALL_DRIVER."'");
				}
			}
		}
		$this->sendFeedback($result);
		return $result;
	}
	
	private function resolveActionCaller($action) {
		$caller = false;
		if(isset($action['caller']) && !empty($action['caller'])) {
			$actionCaller = trim($action['caller']);
			if(PLUGIN_PLACECALL_CALLER_PROPERTY) {
	
			} else {
				$settings = $this->getSettings();
				$settings->set('zarafa/v1/plugins/placecall/caller', $actionCaller);
				$caller = $actionCaller;
			}
		}
		return $caller;
	}
	
	private function resolveActionCallee($action) {
		$callee = false;
		if(isset($action['callee']) && !empty($action['callee'])) {
			$callee = trim($action['callee']);
		}
		return $callee;
	}

	private function getUserCaller() {
		$caller = false;
		$session = $this->getSession();
		$user = $session->getUserName();
		$userEntryid = $session->getUserEntryID();
		$userEntryid || $this->log("ERROR: Unable to retrieve user entry id for user '{$user}'");
		$userEntry = mapi_ab_openentry($session->getAddressbook(), $userEntryid);
		$userEntry || $this->log("ERROR: Unable to retrieve user entry for user '{$user}'");
		$callerProps = mapi_getprops($userEntry, array(PLUGIN_PLACECALL_CALLER_PROPERTY));
		if($callerProps) {
			$callerProp = (isset($callerProps[PLUGIN_PLACECALL_CALLER_PROPERTY]) ? $callerProps[PLUGIN_PLACECALL_CALLER_PROPERTY] : '');
			if($callerProp != '') {
				$caller = $callerProp;
			} else {
				$this->log("ERROR: Caller property not set for user '{$user}'");
			}
		} else {
			$this->log("ERROR: Unable to retrieve user properties for user '{$user}'");
		}
		return $caller;
	}
	
	private function normalizeCallee($callee) {
		$mapped = "";
		if(PLUGIN_PLACECALL_NORMALIZECALLEE) {
			for($n = 0, $strlen = strlen($callee); $n < $strlen && $mapped !== false; $n++) {
				$c = $callee[$n];
				switch($c) {
					case '0':
					case '1':
					case '2':
					case '3':
					case '4':
					case '5':
					case '6':
					case '7':
					case '8':
					case '9':
						$mapped .= $c;
						break;
					case '+':
						$mapped = ($n == 0 ? PLUGIN_PLACECALL_INTPREFIX : false);
						break;
					case ' ':
					case '-':
						break;
					default:
						$mapped = false;
						break;
				}
			}
			if(strlen($mapped) > 0) {
				if(!ereg('^'.PLUGIN_PLACECALL_INTPREFIX, $mapped)) {
					if(!ereg('^'.PLUGIN_PLACECALL_NATPREFIX, $mapped)) {
						$mapped = PLUGIN_PLACECALL_INTPREFIX.PLUGIN_PLACECALL_INTCODE.PLUGIN_PLACECALL_NATCODE.$mapped;
					} else {
						$mapped = PLUGIN_PLACECALL_INTPREFIX.PLUGIN_PLACECALL_INTCODE.substr($mapped,strlen(PLUGIN_PLACECALL_NATPREFIX));
					}
				}
			} else {
				$mapped = false;
			}
		} else {
			$mapped = $callee;
		}
		return $mapped;
	}
	
	private function astspoolPlaceCall($caller, $callee) {
		$this->log("DEBUG: Generating Asterisk call file for caller '{$caller}' and callee '{$callee}'...");
		$callchannel = sprintf(PLUGIN_PLACECALL_ASTSPOOL_CHANNEL, $caller);
		$normalizedCallee = $this->normalizeCallee($callee);
		$call = "Channel: {$callchannel}";
		$call .= "\nExtension: {$normalizedCallee}";
		$call .= "\nCallerID: {$callee}";
		$call .= "\nMaxRetries: ".PLUGIN_PLACECALL_ASTSPOOL_MAXRETRIES;
		$call .= "\nRetryTime: ".PLUGIN_PLACECALL_ASTSPOOL_RETRYTIME;
		$call .= "\nWaitTime: ".PLUGIN_PLACECALL_ASTSPOOL_WAITTIME;
		$call .= "\nContext: ".PLUGIN_PLACECALL_ASTSPOOL_CONTEXT;
		$call .= "\nPriority: ".PLUGIN_PLACECALL_ASTSPOOL_PRIORITY;
		$call .= "\n";
		$this->log($call, false);
		return $this->astspoolCreateCallFile($call);		
	}

	private function astspoolCreateCallFile($call) {
		$status = false;
		$tmppath = (PLUGIN_PLACECALL_ASTSPOOL_TMPPATH !== false ? PLUGIN_PLACECALL_ASTSPOOL_TMPPATH : sys_get_temp_dir());
		$tmppath = realpath($tmppath);
		do {
			$tmpname = $tmppath.DIRECTORY_SEPARATOR.'placecall.'.rand(); 
		} while(file_exists($tmpname));
		if(file_put_contents($tmpname, $call)) {
			if(chmod($tmpname, 0660)) {
				if(chgrp($tmpname, PLUGIN_PLACECALL_ASTSPOOL_CHGGRP)) {
					$callpath = realpath(PLUGIN_PLACECALL_ASTSPOOL_OUTGOINGDIR);
					do {
						$callname = $callpath.DIRECTORY_SEPARATOR.'placecall.'.rand();
					} while(file_exists($callname));
							$this->log($n++);
					if(rename($tmpname, $callname)) {
						$this->log("DEBUG: Call file written to '{$callname}'");
						$status = true;
					} else {
						$this->log("ERROR: Unable to rename temporary file '{$tmpname}' to '{$callname}'");
					}
				} else {
					$this->log("ERROR: Unable to chgrp temporary file '{$tmpname}' to ".PLUGIN_PLACECALL_ASTSPOOL_CHGGRP);
				}
			} else {
				$this->log("ERROR: Unable to chmod temporary file '{$tmpname}'");
			}
		} else {
			$this->log("ERROR: Unable to write temporary file '{$tmpname}'");
		}
		return $status;
	}
	
	private function astmgrPlaceCall($caller,$callee) {
		$status = false;
		$this->log("DEBUG: Orginating call via Asterisk Manager for caller '{$caller}' and callee '{$callee}'...");
		$socket = $this->astmgrLogin();
		if($socket) {
			$status = $this->astmgrOriginate($socket,$caller,$callee);
			$this->astmgrLogoff($socket);			
		}
		return $status;
	}

	private function astmgrLogin() {
		$socket = fsockopen(PLUGIN_PLACECALL_ASTMGR_HOST,PLUGIN_PLACECALL_ASTMGR_PORT,$errno,$errstr,1);
		if($socket) {
			stream_set_timeout($socket,1);
			$loginAction = 'Action: Login';
			$loginAction .= "\r\nUsername: ".PLUGIN_PLACECALL_ASTMGR_USERNAME;
			$loginAction .= "\r\nSecret: ".PLUGIN_PLACECALL_ASTMGR_SECRET;
			$loginAction .= "\r\nEvents: off";
			$loginAction .= "\r\n\r\n";
			$response = $this->astmgrSendRecv($socket,$loginAction);
			if(stripos($response,'Response: Success') === false) {
				$this->log('ERROR: Asterisk Manager login failed');
				fclose($socket);
				$socket = false;
			}
		} else {
			$this->log('ERROR: Connection to '.PLUGIN_PLACECALL_ASTMGR_HOST.':'.PLUGIN_PLACECALL_ASTMGR_PORT."failed with error: ({$errno}) ${errstr}");
		}
		return $socket;
	}
	
	private function astmgrLogoff($socket) {
		$logoffAction = 'Action: Logoff';
		$logoffAction .= "\r\n\r\n";
		$this->astmgrSendRecv($socket,$logoffAction);
		fclose($socket);
	}

	private function astmgrOriginate($socket,$caller,$callee) {
		$callchannel = sprintf(PLUGIN_PLACECALL_ASTMGR_CHANNEL, $caller);
		$normalizedCallee = $this->normalizeCallee($callee);
		$originateAction = 'Action: Originate';
		$originateAction .= "\r\nChannel: {$callchannel}";
		$originateAction .= "\r\nExten: {$normalizedCallee}";
		$originateAction .= "\r\nCallerID: {$callee}";
		$originateAction .= "\r\nPriority: ".PLUGIN_PLACECALL_ASTMGR_PRIORITY;
		$originateAction .= "\r\nTimeout: ".PLUGIN_PLACECALL_ASTMGR_TIMEOUT;
		$originateAction .= "\r\nContext: ".PLUGIN_PLACECALL_ASTMGR_CONTEXT;
		$originateAction .= "\r\nAsync: 1";
		$originateAction .= "\r\n\r\n";
		$response = $this->astmgrSendRecv($socket,$originateAction);
		return stripos($response,'Response: Success') !== false;
	}
	
	private function astmgrSendRecv($socket,$action) {
		$response = false;
		if($socket) {
			$this->log('DEBUG: Sending request');
			$this->log($action,false);
			fputs($socket,$action);
			do {
				$received = fgets($socket);
				if($received) {
					$response .= $received;
				}
				$status = stream_get_meta_data($socket);
			} while($received != "\r\n" && $status['timed_out'] == false);
			$this->log('DEBUG: Received response');
			$this->log($response,false);
		}
		return $response;
	}
	
	private function log($message, $newline=true) {
		if(PLUGIN_PLACECALL_LOGGING_ENABLED) {
			$log = fopen(PLUGIN_PLACECALL_LOGGING_FILE, 'a');
			if($log) {
				fwrite($log, $message);
				if($newline) {
					fwrite($log, "\n");
				}
				fclose($log);
			}
		}
	}
	
	private function getSession() {
		return $GLOBALS['mapisession'];
	}

	private function getOperations() {
		return $GLOBALS["operations"];
	}
	
	private function getBus() {
		return $GLOBALS['bus'];
	}
	
	private function getSettings() {
		return $GLOBALS['settings'];
	}
	
}
?>
