Ext.namespace('Zarafa.plugins.placecall');

/**
 * @class Zarafa.plugins.placecall.PlacecallPlugin
 * @extends Zarafa.core.Plugin
 *
 * This class integrates the plugin into the WebApp and allows
 * the user to place calls to a contact's phone numbers.
 */
Zarafa.plugins.placecall.PlacecallPlugin = Ext.extend(Zarafa.core.Plugin, {

	/**
	 * @constructor
	 * @param {Object} config Configuration object
	 */
	constructor : function (config)
	{
		config = config || {};
		Ext.applyIf(config, {
			name : 'placecall'
		});
		Zarafa.plugins.placecall.PlacecallPlugin.superclass.constructor.call(this, config);
		this.registerInsertionPoint('context.addressbook.contextmenu.actions', this.createPlaceCallAddressBookContextItem, this);
		this.registerInsertionPoint('context.contact.contextmenu.actions', this.createPlaceCallContactContextItem, this);
		this.registerInsertionPoint('context.contact.contactcontentpanel.toolbar.actions', this.createPlaceCallContactButton, this);
		Zarafa.core.data.SharedComponentType.addProperty('plugins.placecall.dialogs.preparecall');
	},

	/**
	 * Creates the place call command entry in the AddressBook context menu.
	 *
	 * @return {Zarafa.core.ui.menu.ConditionalItem} menuItem MenuItem instance
	 */
	createPlaceCallAddressBookContextItem : function()
	{
		return {
			xtype               : 'zarafa.conditionalitem',
			text				: _('Call contact', 'plugin_placecall'),
			iconCls				: 'icon_placecall',
			handler				: function(item) {
				this.placeCallFromAddressBookRecord(item.parentMenu.records[0]);
			},
			singleSelectOnly	: true,
			scope				: this
		};
	},

	/**
	 * Creates the place call command entry in the Contact context menu.
	 *
	 * @return {Zarafa.core.ui.menu.ConditionalItem} menuItem MenuItem instance
	 */
	createPlaceCallContactContextItem : function()
	{
		return {
			xtype               : 'zarafa.conditionalitem',
			text				: _('Call contact', 'plugin_placecall'),
			iconCls				: 'icon_placecall',
			handler				: function(item) {
				this.placeCallFromContactRecord(item.parentMenu.records[0]);
			},
			singleSelectOnly	: true,
			scope				: this
		};
	},

	/**
	 * Creates the place call command button in the Contact toolbar.
	 *
	 * @return {Ext.Button} button Button instance
	 */
	createPlaceCallContactButton : function()
	{
		return {
			xtype               : 'button',
			plugins				: [ 'zarafa.recordcomponentupdaterplugin' ],
			iconCls				: 'icon_placecall',
			overflowText		: _('Call contact', 'plugin_placecall'),
			tooltip : {
				title : _('Call contact', 'plugin_placecall'),
				text : _('Place a call to one of this recipient\'s phone numbers.', 'plugin_placecall')
			},
			handler				: function(btn) {
				this.placeCallFromContactRecord(btn.record);
			},
			scope				: this,
			update				: function(record, resetContent) {
				this.record = record;
			}
		};
	},

	/**
	 * Invoke the place call dialog for the currently selected AddressBook entry.
	 *
	 * @param {Object} record The selected AddressBook record.
	 */
	placeCallFromAddressBookRecord : function(record)
	{
		container.getRequest().singleRequest('placecallmodule', 'preparefromaddressbook', {
			entryid : record.get('entryid'),
			profile : 'placecall'
		}, new Zarafa.plugins.placecall.data.PrepareCallRecordResponseHandler());
	},

	/**
	 * Invoke the place call dialog for the currently selected Contact entry.
	 *
	 * @param {Object} record The selected Contact record.
	 */
	placeCallFromContactRecord : function(record)
	{
		container.getRequest().singleRequest('placecallmodule', 'preparefromcontact', {
			entryid : record.get('entryid'),
			parent_entryid : record.get('parent_entryid'),
			store_entryid : record.get('store_entryid'),
			profile : 'placecall'
		}, new Zarafa.plugins.placecall.data.PrepareCallRecordResponseHandler());
	},
	
	bidSharedComponent : function(type, record)
	{
		var bid = -1;
		
		switch(type)
		{
			case Zarafa.core.data.SharedComponentType['plugins.placecall.dialogs.preparecall']:
				bid = 1;
				break;
		}
		return bid;
	},
	
	getSharedComponent : function(type, record)
	{
		var component = Zarafa.plugins.placecall.dialogs.PrepareCallContentPanel;

		return component;
	}

});

Zarafa.onReady(function() {
	if(container.getSettingsModel().get('zarafa/v1/plugins/placecall/enable') === true) {
		container.registerPlugin(new Zarafa.plugins.placecall.PlacecallPlugin());
	}
});
Ext.namespace('Zarafa.plugins.placecall.data');

/**
 * @class Zarafa.plugins.placecall.data.PrepareCallRecordResponseHandler
 * @extends Zarafa.core.data.AbstractResponseHandler
 *
 * Response handler dealing with prepare call data returned by the backend in order to choose the call details.
 */
Zarafa.plugins.placecall.data.PrepareCallRecordResponseHandler = Ext.extend(Zarafa.core.data.AbstractResponseHandler, {

	/**
	 * Evaluates the returned data object and either shows a message box in case the information is not
	 * sufficient or invokes the PrepareCallDialog to let the user initiate the call.
	 * @param {Object} response Object contained the response data.
	 */
	doItem : function(response) {
		var message = response.message;
		
		if(Ext.isEmpty(message)) {
			var componentType = Zarafa.core.data.SharedComponentType['plugins.placecall.dialogs.preparecall'];
			var records = {};
			var config = {
					params	: {
							callerReadOnly	: response.callerReadOnly,
							caller			: response.caller,
							callees			: response.callees
						},
					modal	: true
				};
			
			Zarafa.core.data.UIFactory.openLayerComponent(componentType, records, config);
		} else {
			Ext.MessageBox.show({
				title   : _('Warning'),
				msg     : _(message, 'plugin_placecall'),
				icon    : Ext.MessageBox.WARNING,
				buttons : Ext.MessageBox.OK
			});
		}
	}
});

Ext.reg('placecall.preparecallrecordresponsehandler', Zarafa.plugins.placecall.data.PrepareCallRecordResponseHandler);
Ext.namespace('Zarafa.plugins.placecall.dialogs');

/**
 * @class Zarafa.plugins.placecall.dialogs.PrepareCallDialog
 * @extends Zarafa.core.ui.Dialog
 *
 * Dialog to let the user choose the actual call parameters.
 */
Zarafa.plugins.placecall.dialogs.PrepareCallContentPanel = Ext.extend(Zarafa.core.ui.ContentPanel, {

	/**
	 * @constructor
	 * @param config Configuration structure
	 */
	constructor : function(config)
	{
		config = config || {};

		var params = config.params;
		var callerReadOnly = true;
		var caller = '';
		var callees = [];
		
		if(!Ext.isEmpty(params)) {
			callerReadOnly = params.callerReadOnly;
			caller = params.caller;
			callees = params.callees;
		}
		
		var calleeRadios = [];
		
		for(var key in callees) {
			calleeRadios.push({
				xtype: 'radio',
	        	boxLabel: _(key, 'plugin_placecall') + ': ' + callees[key],
	        	name: "callee",
        		inputValue : callees[key]
			});
		}

		Ext.applyIf(config, {
			title  : _('Call contact', 'plugin_placecall'),
			layout : 'fit',
			width  : 400,
			height : 120,
			items  : [{
				xtype  : 'form',
				layout : 'form',
				border : false,
				ref    : 'form',
				items  : [{
						xtype		: 'textfield',
						name		: 'caller',
						value		: caller,
						allowBlank	: false,
						readOnly	: callerReadOnly,
						fieldLabel	: _('Call via', 'plugin_placecall'),
						anchor		: '100%'
					}, {
						xtype : 'radiogroup',
						name : 'callee',
					    columns: 1,
						fieldLabel : _('Call number', 'plugin_placecall'),
						value : calleeRadios[0].inputValue,
					    items : calleeRadios
					}
				],
				buttons: [{
						text: _('Ok'),
						handler: this.onOk,
						scope: this
					},
					{
						text: _('Cancel'),
						handler: this.onCancel,
						scope: this
					}
				]
			}]
		});

		//Call superclass constructor
		Zarafa.plugins.placecall.dialogs.PrepareCallContentPanel.superclass.constructor.call(this, config);
	},
	
	onOk : function()
	{
		var form = this.form.getForm();
		
		if(form.isValid()) {
			var values = form.getValues();
			
			container.getRequest().singleRequest('placecallmodule', 'placecall', {
				caller : values['caller'],
				callee : values['callee'],
				profile : 'placecall'
			});
			this.close();
		}
	},
	
	onCancel : function()
	{
		this.close();
	}

});

//Register the dialog xtype
Ext.reg('placecall.preparecalldialog', Zarafa.plugins.placecall.dialogs.PrepareCallContentPanel);
