<?php
/*
 * Copyright 2012-2013 Holger de Carne
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License, version 3, 
 * as published by the Free Software Foundation with the following additional 
 * term according to sec. 7:
 *  
 * According to sec. 7 of the GNU Affero General Public License, version
 * 3, the terms of the AGPL are supplemented with the following terms:
 * 
 * "Zarafa" is a registered trademark of Zarafa B.V. The licensing of
 * the Program under the AGPL does not imply a trademark license.
 * Therefore any rights, title and interest in our trademarks remain
 * entirely with us.
 * 
 * However, if you propagate an unmodified version of the Program you are
 * allowed to use the term "Zarafa" to indicate that you distribute the
 * Program. Furthermore you may use our trademarks where it is necessary
 * to indicate the intended purpose of a product or service provided you
 * use it in accordance with honest practices in industrial or commercial
 * matters.  If you want to propagate modified versions of the Program
 * under the name "Zarafa" or "Zarafa Server", you may only do so if you
 * have a written permission by Zarafa B.V. (to acquire a permission
 * please contact Zarafa at trademark@zarafa.com).
 * 
 * The interactive user interface of the software displays an attribution
 * notice containing the term "Zarafa" and/or the logo of Zarafa.
 * Interactive user interfaces of unmodified and modified versions must
 * display Appropriate Legal Notices according to sec. 5 of the GNU
 * Affero General Public License, version 3, when you propagate
 * unmodified or modified versions of the Program. In accordance with
 * sec. 7 b) of the GNU Affero General Public License, version 3, these
 * Appropriate Legal Notices must retain the logo of Zarafa or display
 * the words "Initial Development by Zarafa" if the display of the logo
 * is not reasonably feasible for technical reasons. The use of the logo
 * of Zarafa in Legal Notices is allowed for unmodified and modified
 * versions of the software.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero General Public License for more details.
 *  
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

require_once('api/placecall_api.php');

/**
 * Placecall Plugin module
 *
 * This class processes the call invokations from the client and
 * triggers the PBX system accordingly.
 */
class PlacecallModule extends Module {

	/* Phone properties for callee selection */
	private static $phonePropKeys = array(
				'business'  => PR_BUSINESS_TELEPHONE_NUMBER,
				'business2' => PR_BUSINESS2_TELEPHONE_NUMBER,
				'home'      => PR_HOME_TELEPHONE_NUMBER,
				'home2'     => PR_HOME2_TELEPHONE_NUMBER,
				'primary'   => PR_PRIMARY_TELEPHONE_NUMBER,
				'callback'  => PR_CALLBACK_TELEPHONE_NUMBER,
				'mobile'    => PR_MOBILE_TELEPHONE_NUMBER,
				'radio'     => PR_RADIO_TELEPHONE_NUMBER,
				'car'       => PR_CAR_TELEPHONE_NUMBER,
				'other'     => PR_OTHER_TELEPHONE_NUMBER,
				'pager'     => PR_PAGER_TELEPHONE_NUMBER,
				'assistant' => PR_ASSISTANT_TELEPHONE_NUMBER
			);

	/**
	 * Constructor
	 * @param int $id unique id.
	 * @param array $data list of all actions.
	 */
	public function PlacecallModule($id, $data) {
		parent::Module($id, $data);
	}

	/**
	 * Process the incoming events that were fired by the client.
	 * @return boolean True if everything was processed correctly.
	 */
	public function execute() {
		$result = false;
		foreach($this->data as $actionType => $actionData) {
			if(isset($actionType)) {
				$this->log("DEBUG: enter::{$actionType}");
				try {
					switch($actionType) {
						case 'preparefromaddressbook':
							$result = $this->prepareFromAddressBook($actionData);
							break;
						case 'preparefromcontact':
							$result = $this->prepareFromContact($actionData);
							break;
						case 'placecall':
							$result = $this->placeCall($actionData);
							break;
						default:
							$this->handleUnknownActionType($actionType);
					}
				} catch (MAPIException $e) {
					$this->log("ERROR: Exception '{$e}'");
					$this->sendFeedback(false, $this->errorDetailsFromException($e));
				}
				$resultString = ($result ? 'true' : 'false');
				$this->log("DEBUG: exit::{$actionType} (result={$resultString})");
			}
		}
		return $result;
	}
	
	/**
	 * Prepare call based upon a AddressBook entry.
	 * @param mixed $action The data that holds the action request data.
	 * @return boolean True if the action succeeded.
	 **/
	private function prepareFromAddressBook($action) {
		$entryid = $this->getActionEntryID($action);
		$entryid || $this->log('ERROR: Unable to determine entry id');
		$addressbook = mapi_openaddressbook($this->getSession()->getSession());
		$addressbook || $this->log('ERROR: Unable to open addressbook');
		$entry = mapi_ab_openentry($addressbook, $entryid);
		$entry || $this->log('ERROR: Unable to get addressbook entry');
		return $this->prepareResponse($entry);
	}
	
	/**
	 * Prepare call based upon a Contact entry.
	 * @param mixed $action The data that holds the action request data.
	 * @return boolean True if the action succeeded.
	 **/
	private function prepareFromContact($action) {
		$store = $this->getActionStore($action);
		$store || $this->log('ERROR: Unable to open store'); 
		$entryid = $this->getActionEntryID($action);
		$entryid || $this->log('ERROR: Unable to determine entry id');
		$entry = mapi_msgstore_openentry($store, $entryid);
		$entry || $this->log('ERROR: Unable to get contact entry');
		return $this->prepareResponse($entry);
	}

	/**
	 * Prepare and send the response object to the client.
	 * @param mixed $entry The AddressBook or Contact entry containing the callee information. 
	 * @return boolean True if action succeeded.
	 */
	private function prepareResponse($entry) {
		$result = false;
		$response = array();
		$caller = $this->prepareCaller();
		if($caller !== false) {
			$callees = $this->prepareCallees($entry);
			if($callees !== false) {
				if(count($callees) > 0) {
					$response['callerReadOnly'] = PLUGIN_PLACECALL_CALLER_PROPERTY !== false;
					$response['caller'] = $caller;
					$response['callees'] = $callees;
					$result = true;
				} else {
					$response['message'] = 'Contact has no telephone numbers assigned.';
				}
			} else {
				$response['message'] = 'Unable to retrieve the contact information.';
			}
		} else {
			$response['message'] = 'Unable to determine your caller id. Please contact your administrator.';
		}
		$this->addActionData("item", $response);
		$bus = $this->getBus();
		$bus->addData($this->getResponseData());
		return $result;
	}
	
	private function prepareCaller() {
		$caller = false;
		if(PLUGIN_PLACECALL_CALLER_PROPERTY) {
			$caller = $this->getUserCaller();
		} else {
			$settings = $this->getSettings();
			$caller = $settings->get('zarafa/v1/plugins/placecall/caller', '');
		}
		return $caller;
	}
	
	private function prepareCallees($contact) {
		$callees = false;
		$contactProps = mapi_getprops($contact, self::$phonePropKeys);
		$contactProps || $this->log('ERROR: Unable to retrieve contact properties');
		if($contactProps) {
			$callees = array();
			$this->log('DEBUG: Collecting callees for contact:');
			foreach(self::$phonePropKeys as $phonePropKeyId => $phonePropKey) {
				$callee = MAPI_E_NOT_FOUND;
				if(isset($contactProps[$phonePropKey])) {
					$callee = $contactProps[$phonePropKey];
				}
				if($callee !== MAPI_E_NOT_FOUND && ($callee = trim($callee)) != '') {
					$callees[$phonePropKeyId] = $this->normalize($callee);
					$this->log("DEBUG: Collecting phone number '{$phonePropKeyId}' = '{$callee}'");
				} else {
					$this->log("DEBUG: Skipping undefined phone number '{$phonePropKeyId}'");
				}
			}
		}
		return $callees;
	}
	
	/**
	 * Place a call for the selected numbers.
	 * @param mixed $action The data that holds the action request data.
	 * @return boolean True if the action succeeded.
	 **/
	private function placeCall($action) {
		$result = false;
		$caller = $this->resolveActionCaller($action);
		$this->log("DEBUG: caller = '{$caller}'");
		if($caller) {
			$callee = $this->resolveActionCallee($action);
			$this->log("DEBUG: callee = '{$callee}'");
			if($callee) {
				$api = PlaceCallAPI::init($this, PLUGIN_PLACECALL_API);
				$result = $api->placeCall($caller, $callee);
			}
		}
		$this->sendFeedback($result);
		return $result;
	}
	
	private function resolveActionCaller($action) {
		$caller = false;
		if(isset($action['caller']) && !empty($action['caller'])) {
			$actionCaller = trim($action['caller']);
			if(PLUGIN_PLACECALL_CALLER_PROPERTY) {
				$caller = $this->getUserCaller();
			} else {
				$settings = $this->getSettings();
				$settings->set('zarafa/v1/plugins/placecall/caller', $actionCaller);
				$caller = $actionCaller;
			}
		}
		return $caller;
	}
	
	private function resolveActionCallee($action) {
		$callee = false;
		if(isset($action['callee']) && !empty($action['callee'])) {
			$callee = trim($action['callee']);
		}
		return $callee;
	}

	private function getUserCaller() {
		$caller = false;
		$session = $this->getSession();
		$user = $session->getUserName();
		$userEntryid = $session->getUserEntryID();
		$userEntryid || $this->log("ERROR: Unable to retrieve user entry id for user '{$user}'");
		$userEntry = mapi_ab_openentry($session->getAddressbook(), $userEntryid);
		$userEntry || $this->log("ERROR: Unable to retrieve user entry for user '{$user}'");
		$callerProps = mapi_getprops($userEntry, array(PLUGIN_PLACECALL_CALLER_PROPERTY));
		if($callerProps) {
			$callerProp = (isset($callerProps[PLUGIN_PLACECALL_CALLER_PROPERTY]) ? $callerProps[PLUGIN_PLACECALL_CALLER_PROPERTY] : '');
			if($callerProp != '') {
				$caller = $callerProp;
			} else {
				$this->log("ERROR: Caller property not set for user '{$user}'");
			}
		} else {
			$this->log("ERROR: Unable to retrieve user properties for user '{$user}'");
		}
		return $caller;
	}
	
	private function normalize($cid) {
		$cidN = preg_replace('/[^\+0-9]/', '', $cid);
		if(strpos($cidN, '+') === 0) {
			$cidN = PLUGIN_PLACECALL_INTPREFIX.substr($cidN, 1);
		} elseif(strpos($cidN, PLUGIN_PLACECALL_INTPREFIX) !== 0) {
			if(strpos($cidN, PLUGIN_PLACECALL_NATPREFIX) === 0) {
				$cidN = PLUGIN_PLACECALL_INTPREFIX.PLUGIN_PLACECALL_INTCODE.substr($cidN, strlen(PLUGIN_PLACECALL_NATPREFIX));
			} else {
				$cidN = PLUGIN_PLACECALL_INTPREFIX.PLUGIN_PLACECALL_INTCODE.PLUGIN_PLACECALL_NATCODE.$cidN;
			}
		}
		return $cidN;
	}
	
	public function log($message, $newline=true) {
		if(PLUGIN_PLACECALL_LOGGING_ENABLED) {
			$log = fopen(PLUGIN_PLACECALL_LOGGING_FILE, 'a');
			if($log) {
				fwrite($log, $message);
				if($newline) {
					fwrite($log, "\n");
				}
				fclose($log);
			}
		}
	}
	
	private function getSession() {
		return $GLOBALS['mapisession'];
	}

	private function getOperations() {
		return $GLOBALS["operations"];
	}
	
	private function getBus() {
		return $GLOBALS['bus'];
	}
	
	private function getSettings() {
		return $GLOBALS['settings'];
	}
	
}

?>
