<?php
/*
 * Copyright 2012-2013 Holger de Carne
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License, version 3, 
 * as published by the Free Software Foundation with the following additional 
 * term according to sec. 7:
 *  
 * According to sec. 7 of the GNU Affero General Public License, version
 * 3, the terms of the AGPL are supplemented with the following terms:
 * 
 * "Zarafa" is a registered trademark of Zarafa B.V. The licensing of
 * the Program under the AGPL does not imply a trademark license.
 * Therefore any rights, title and interest in our trademarks remain
 * entirely with us.
 * 
 * However, if you propagate an unmodified version of the Program you are
 * allowed to use the term "Zarafa" to indicate that you distribute the
 * Program. Furthermore you may use our trademarks where it is necessary
 * to indicate the intended purpose of a product or service provided you
 * use it in accordance with honest practices in industrial or commercial
 * matters.  If you want to propagate modified versions of the Program
 * under the name "Zarafa" or "Zarafa Server", you may only do so if you
 * have a written permission by Zarafa B.V. (to acquire a permission
 * please contact Zarafa at trademark@zarafa.com).
 * 
 * The interactive user interface of the software displays an attribution
 * notice containing the term "Zarafa" and/or the logo of Zarafa.
 * Interactive user interfaces of unmodified and modified versions must
 * display Appropriate Legal Notices according to sec. 5 of the GNU
 * Affero General Public License, version 3, when you propagate
 * unmodified or modified versions of the Program. In accordance with
 * sec. 7 b) of the GNU Affero General Public License, version 3, these
 * Appropriate Legal Notices must retain the logo of Zarafa or display
 * the words "Initial Development by Zarafa" if the display of the logo
 * is not reasonably feasible for technical reasons. The use of the logo
 * of Zarafa in Legal Notices is allowed for unmodified and modified
 * versions of the software.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero General Public License for more details.
 *  
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

/**
 * Spamtrain Plugin module
 *
 * This class processes the training commands from the client and
 * invokes the corresponding system calls on the spam filter
 */
class SpamtrainModule extends Module {

	/**
	 * Constructor
	 * @param int $id unique id.
	 * @param array $data list of all actions.
	 */
	public function SpamtrainModule($id, $data) {
		parent::Module($id, $data);
	}

	/**
	 * Process the incoming events that were fired by the client.
	 * @return boolean True if everything was processed correctly.
	 */
	public function execute() {
		$result = false;
		foreach($this->data as $actionType => $actionData) {
			if(isset($actionType)) {
				$this->log("DEBUG: enter::{$actionType}");
				try {
					switch($actionType) {
						case 'trainspam':
							$result = $this->trainSpam($actionData);
							break;
						case 'trainham':
							$result = $this->trainHam($actionData);
							break;
						default:
							$this->handleUnknownActionType($actionType);
					}
				} catch (MAPIException $e) {
					$this->log("ERROR: Exception '{$e}'");
					$this->sendFeedback(false, $this->errorDetailsFromException($e));
				}
				$resultString = ($result ? 'true' : 'false');
				$this->log("DEBUG: exit::{$actionType} (result={$resultString})");
			}
		}
		return $result;
	}

	/**
	 * Retrieve the selected emails and feed them to spam train command.
	 * @param mixed $action The data that holds the action request data.
	 * @return boolean True if the action succeeded.
	 **/
	private function trainSpam($action) {
		$result = false;
		if(defined('PLUGIN_SPAMTRAIN_TRAIN_SPAM_CMD')) {
			$store = $this->getActionStore($action);
			$entryid =$this->getActionEntryID($action);
			$result = $this->invokeTrainCmd(PLUGIN_SPAMTRAIN_TRAIN_SPAM_CMD, $store, $entryid);
			if($result) {
				$parententryid = $this->getActionParentEntryID($action);
				$this->log("DEBUG: Deleting message...");
				$notifyProps = array();
				$notifyProps[PR_PARENT_ENTRYID] = $parententryid;
				$notifyProps[PR_ENTRYID] = $entryid;
				$storeprops = mapi_getprops($store, array(PR_ENTRYID));
				$notifyProps[PR_STORE_ENTRYID] = $storeprops[PR_ENTRYID];
				$operations = $this->getOperations();
				$result = $operations->deleteMessages($store, $parententryid, $entryid);
				if($result) {
					$bus = $this->getBus();
					$bus->notify(bin2hex($parententryid), TABLE_DELETE, $notifyProps);
					$this->log("DEBUG: Message deleted");
				} else {
					$this->log("ERROR: Unable to delete message");
				}
			}
		}
		$this->sendFeedback($result);
		return $result;
	}
	
	/**
	 * Retrieve the selected emails and feed them to ham train command.
	 * @param mixed $action The data that holds the action request data.
	 * @return boolean True if the action succeeded.
	 **/
	private function trainHam($action) {
		$result = false;
		if(defined('PLUGIN_SPAMTRAIN_TRAIN_HAM_CMD')) {
			$store = $this->getActionStore($action);
			$entryid =$this->getActionEntryID($action);
			$result = $this->invokeTrainCmd(PLUGIN_SPAMTRAIN_TRAIN_HAM_CMD, $store, $entryid);
		}
		$this->sendFeedback($result);
		return $result;
	}

	private function invokeTrainCmd($cmdPattern, $store, $entryid) {
		$result = false;
		$session = $this->getSession();
		$session !== false || $this->log("ERROR: Unable to open session");
		$user = $session->getUserName();
		$cmd = str_replace('%u', $user, $cmdPattern);
		$this->log("DEBUG: Invoking command '{$cmd}'...");
		$addrbook = $session->getAddressbook();
		$addrbook !== false || $this->log("ERROR: Unable to open address book");
		$message = mapi_msgstore_openentry($store, $entryid);
		$message !== false || $this->log("ERROR: Unable to open message");
		$props = mapi_getprops($message, array(PR_TRANSPORT_MESSAGE_HEADERS));
		$props !== false || $this->log("ERROR: Unable to retrieve message properties");
		$headers = str_replace("\r\n", "\n", $props[PR_TRANSPORT_MESSAGE_HEADERS]);
		$headers || $this->log("ERROR: Unable to retrieve message headers");
		$stream = mapi_inetmapi_imtoinet($session->getSession(), $addrbook, $message, array());
		$stream !== false || $this->log("ERROR: Unable to open message stream"); 
		if($headers && $stream !== false ) {
			$pipe = popen($cmd, 'w');
			if($pipe) {
				$this->log("DEBUG: Streaming message to command:");
				fwrite($pipe, $headers);
				$this->log($headers, false);
				$headerSkipped = false;
				$trailingNewline = false;			
				while(($read = mapi_stream_read($stream, 4096))) {
					$read = str_replace("\r\n", "\n", $read);
					if($headerSkipped) {
						fwrite($pipe, $read);
						$this->log($read, false);
					} else {
						if($trailingNewline && substr($read, 0, 1) == "\n") {
							fwrite($pipe, "\n");
							$this->log("\n", false);
							fwrite($pipe, $read);
							$this->log($read, false);
							$headerSkipped = true;
						} else if(($bodyIndex = strpos($read, "\n\n")) !== false) {
							$bodyStart = substr($read, $bodyIndex);
							fwrite($pipe, $bodyStart);
							$this->log($bodyStart, false);
							$headerSkipped = true;
						} else {
							$trailingNewline =  substr(strlen($read)-1) == "\n";
						}
					}
				}
				$status = pclose($pipe);
				if($status === 0) {
					$this->log("DEBUG: Command '{$cmd}' finished with status {$status}");
					$result = true;
				} else {
					$this->log("ERROR: Command '{$cmd}' finished with status {$status}");
				}
			} else {
				$this->log("ERROR: Unable to execute command '{$cmd}'");
			}
		} else {
			
		}
		return $result;
	} 
	
	private function log($message, $newline=true) {
		if(PLUGIN_SPAMTRAIN_LOGGING_ENABLED) {
			$log = fopen(PLUGIN_SPAMTRAIN_LOGGING_FILE, 'a');
			if($log) {
				fwrite($log, $message);
				if($newline) {
					fwrite($log, "\n");
				}
				fclose($log);
			}
		}
	}
	
	private function getSession() {
		return $GLOBALS['mapisession'];
	}

	private function getOperations() {
		return $GLOBALS["operations"];
	}
	
	private function getBus() {
		return $GLOBALS['bus'];
	}
	
}
?>
