Ext.namespace('Zarafa.plugins.placecall');

/**
 * @class Zarafa.plugins.placecall.ABOUT
 * @extends String
 *
 * The copyright string holding the copyright notice for the placecall Plugin.
 */
Zarafa.plugins.placecall.ABOUT = ""
	+ "<p>Copyright (C) 2012-2013  Holger de Carne</p>"

	+ "<p>This program is free software: you can redistribute it and/or modify "
	+ "it under the terms of the GNU Affero General Public License as "
	+ "published by the Free Software Foundation, either version 3 of the "
	+ "License, or (at your option) any later version.</p>"

	+ "<p>This program is distributed in the hope that it will be useful, "
	+ "but WITHOUT ANY WARRANTY; without even the implied warranty of "
	+ "MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the "
	+ "GNU Affero General Public License for more details.</p>"

	+ "<p>You should have received a copy of the GNU Affero General Public License "
	+ "along with this program.  If not, see <a href=\"http://www.gnu.org/licenses/\" target=\"_blank\">http://www.gnu.org/licenses/</a>.</p>"

	+ "<hr />"

	+ "<p>The plugin icons are subject to the <a href=\"http://creativecommons.org/licenses/by/3.0/\" target=\"_blank\">Creative Commons Attribution 3.0 License</a>."
	+ "See <a href=\"http://pc.de/icons/\" target=\"_blank\">http://pc.de/icons/</a> for further details.</p>";
Ext.namespace('Zarafa.plugins.placecall');

/**
 * @class Zarafa.plugins.placecall.PlacecallPlugin
 * @extends Zarafa.core.Plugin
 *
 * This class integrates the plugin into the WebApp and allows
 * the user to place calls to a contact's phone numbers.
 */
Zarafa.plugins.placecall.PlacecallPlugin = Ext.extend(Zarafa.core.Plugin, {

	/**
	 * @constructor
	 * @param {Object} config Configuration object
	 */
	constructor : function (config)
	{
		config = config || {};
		Ext.applyIf(config, {
			name : 'placecall',
			displayName : _('Place call plugin'),
			about : Zarafa.plugins.placecall.ABOUT
		});
		Zarafa.plugins.placecall.PlacecallPlugin.superclass.constructor.call(this, config);
		if(container.getSettingsModel().get('zarafa/v1/plugins/placecall/enableDirectCall') === true) {
			this.registerInsertionPoint('main.toolbar.actions.last', this.createPlaceCallMainToolBarButton, this);
		}
		this.registerInsertionPoint('context.addressbook.contextmenu.actions', this.createPlaceCallAddressBookContextItem, this);
		this.registerInsertionPoint('context.contact.contextmenu.actions', this.createPlaceCallContactContextItem, this);
		this.registerInsertionPoint('context.contact.contactcontentpanel.toolbar.actions', this.createPlaceCallContactButton, this);
		Zarafa.core.data.SharedComponentType.addProperty('plugins.placecall.dialogs.preparecall');
	},

	/**
	 * Creates the place call button in the Main Toolbar.
	 *
	 * @return {Ext.Button} button Button instance
	 */
	createPlaceCallMainToolBarButton : function()
	{
		return  [{
			xtype		: 'button',
			scale		: 'large',
			title		: _('Call number', 'plugin_placecall'),
			overflowText: _('Call number', 'plugin_placecall'),
			tooltip		: _('Place a direct call to an individual number.', 'plugin_placecall'),
			iconCls		: 'icon_placecall_large',
			handler		: this.placeCallFromInput,
			scope		: this
		}];
	},
	
	/**
	 * Creates the place call command entry in the AddressBook context menu.
	 *
	 * @return {Zarafa.core.ui.menu.ConditionalItem} menuItem MenuItem instance
	 */
	createPlaceCallAddressBookContextItem : function()
	{
		return {
			xtype               : 'zarafa.conditionalitem',
			text				: _('Call contact', 'plugin_placecall'),
			iconCls				: 'icon_placecall',
			handler				: function(item) {
				this.placeCallFromAddressBookRecord(item.parentMenu.records[0]);
			},
			singleSelectOnly	: true,
			scope				: this
		};
	},

	/**
	 * Creates the place call command entry in the Contact context menu.
	 *
	 * @return {Zarafa.core.ui.menu.ConditionalItem} menuItem MenuItem instance
	 */
	createPlaceCallContactContextItem : function()
	{
		return {
			xtype               : 'zarafa.conditionalitem',
			text				: _('Call contact', 'plugin_placecall'),
			iconCls				: 'icon_placecall',
			handler				: function(item) {
				this.placeCallFromContactRecord(item.parentMenu.records[0]);
			},
			singleSelectOnly	: true,
			scope				: this
		};
	},

	/**
	 * Creates the place call command button in the Contact toolbar.
	 *
	 * @return {Ext.Button} button Button instance
	 */
	createPlaceCallContactButton : function()
	{
		return {
			xtype               : 'button',
			plugins				: [ 'zarafa.recordcomponentupdaterplugin' ],
			iconCls				: 'icon_placecall',
			overflowText		: _('Call contact', 'plugin_placecall'),
			tooltip : {
				title : _('Call contact', 'plugin_placecall'),
				text : _('Place a call to one of this recipient\'s phone numbers.', 'plugin_placecall')
			},
			handler				: function(btn) {
				this.placeCallFromContactRecord(btn.record);
			},
			scope				: this,
			update				: function(record, resetContent) {
				this.record = record;
			}
		};
	},

	/**
	 * Invoke the place call dialog for an input based call.
	 */
	placeCallFromInput : function()
	{
		container.getRequest().singleRequest('placecallmodule', 'preparefrominput', {
			profile : 'placecall'
		}, new Zarafa.plugins.placecall.data.PlaceCallResponseHandler());
	},
	
	/**
	 * Invoke the place call dialog for the currently selected AddressBook entry.
	 *
	 * @param {Object} record The selected AddressBook record.
	 */
	placeCallFromAddressBookRecord : function(record)
	{
		container.getRequest().singleRequest('placecallmodule', 'preparefromaddressbook', {
			entryid : record.get('entryid'),
			profile : 'placecall'
		}, new Zarafa.plugins.placecall.data.PlaceCallResponseHandler());
	},

	/**
	 * Invoke the place call dialog for the currently selected Contact entry.
	 *
	 * @param {Object} record The selected Contact record.
	 */
	placeCallFromContactRecord : function(record)
	{
		container.getRequest().singleRequest('placecallmodule', 'preparefromcontact', {
			entryid : record.get('entryid'),
			parent_entryid : record.get('parent_entryid'),
			store_entryid : record.get('store_entryid'),
			profile : 'placecall'
		}, new Zarafa.plugins.placecall.data.PlaceCallResponseHandler());
	},
	
	bidSharedComponent : function(type, record)
	{
		var bid = -1;
		
		switch(type)
		{
			case Zarafa.core.data.SharedComponentType['plugins.placecall.dialogs.preparecall']:
				bid = 1;
				break;
		}
		return bid;
	},
	
	getSharedComponent : function(type, record)
	{
		var component = Zarafa.plugins.placecall.dialogs.PrepareCallContentPanel;

		return component;
	}

});

Zarafa.onReady(function() {
	if(container.getSettingsModel().get('zarafa/v1/plugins/placecall/enable') === true) {
		container.registerPlugin(new Zarafa.plugins.placecall.PlacecallPlugin());
	}
});
Ext.namespace('Zarafa.plugins.placecall.data');

/**
 * @class Zarafa.plugins.placecall.data.PlaceCallResponseHandler
 * @extends Zarafa.core.data.AbstractResponseHandler
 *
 * Response handler dealing with all kinds of responses.
 */
Zarafa.plugins.placecall.data.PlaceCallResponseHandler = Ext.extend(Zarafa.core.data.AbstractResponseHandler, {

	/**
	 * Invoked on success feedback.
	 * Does nothing at the current point in time.
	 * @param {Object} response Object contained the response data.
	 */
	doSuccess : function(response) {
		var callee = response.callee;
		
		container.getNotifier().notify('info.calling', _('Calling', 'plugin_placecall'), callee);		
	},
	
	/**
	 * Invoked on error feedback.
	 * Shows a error popup for the received messages.
	 * @param {Object} response Object contained the response data.
	 */
	doError : function(response) {
		var message = response.message;
		
		Ext.MessageBox.show({
			title   : _('Error'),
			msg     : _(message, 'plugin_placecall'),
			icon    : Ext.MessageBox.ERROR,
			buttons : Ext.MessageBox.OK
		});
	},
	
	/**
	 * Invoked during call preparation. Displays the PrepareCallDialog to let the user prepare and initiate the call.
	 * @param {Object} response Object contained the response data.
	 */
	doPrepare : function(response) {
		var componentType = Zarafa.core.data.SharedComponentType['plugins.placecall.dialogs.preparecall'];
		var records = {};
		var config = {
				params	: {
						callerReadOnly	: response.callerReadOnly,
						caller			: response.caller,
						calleesReadOnly	: response.calleesReadOnly,
						callees			: response.callees
					},
				modal	: true
			};

		Zarafa.core.data.UIFactory.openLayerComponent(componentType, records, config);
	}
	
});

Ext.reg('placecall.placecallresponsehandler', Zarafa.plugins.placecall.data.PlaceCallResponseHandler);
Ext.namespace('Zarafa.plugins.placecall.dialogs');

/**
 * @class Zarafa.plugins.placecall.dialogs.PrepareCallDialog
 * @extends Zarafa.core.ui.Dialog
 *
 * Dialog to let the user choose the actual call parameters.
 */
Zarafa.plugins.placecall.dialogs.PrepareCallContentPanel = Ext.extend(Zarafa.core.ui.ContentPanel, {

	/**
	 * @constructor
	 * @param config Configuration structure
	 */
	constructor : function(config)
	{
		config = config || {};

		var params = config.params;
		var callerReadOnly = true;
		var caller = '';
		var calleesReadOnly = true;
		var callees = [];
		
		if(!Ext.isEmpty(params)) {
			callerReadOnly = params.callerReadOnly;
			caller = params.caller;
			calleesReadOnly = params.calleesReadOnly;
			callees = params.callees;
		}
		
		var calleeItem = {};
		
		if(calleesReadOnly) {
			var calleeRadios = [];

			for(var key in callees) {
				calleeRadios.push({
					xtype: 'radio',
					boxLabel: _(key, 'plugin_placecall') + ': ' + callees[key],
					name: "callee",
					inputValue : callees[key]
				});
			}
			calleeItem = {
				xtype : 'radiogroup',
				name : 'callee',
				columns: 1,
				fieldLabel : _('Call number', 'plugin_placecall'),
				value : calleeRadios[0].inputValue,
				items : calleeRadios
			}; 
		} else {
			calleeItem = {
				xtype		: 'textfield',
				name		: 'callee',
				value		: callees[0],
				allowBlank	: false,
				fieldLabel	: _('Call number', 'plugin_placecall'),
				anchor		: '100%'
			}
		}
		Ext.applyIf(config, {
			title  : _('Call contact', 'plugin_placecall'),
			layout : 'fit',
			width  : 400,
			height : 120,
			items  : [{
				xtype  : 'form',
				layout : 'form',
				border : false,
				ref    : 'form',
				items  : [{
						xtype		: 'textfield',
						name		: 'caller',
						value		: caller,
						allowBlank	: false,
						readOnly	: callerReadOnly,
						fieldLabel	: _('Call via', 'plugin_placecall'),
						anchor		: '100%'
					}, calleeItem
				],
				buttons: [{
						text: _('Ok'),
						handler: this.onOk,
						scope: this
					},
					{
						text: _('Cancel'),
						handler: this.onCancel,
						scope: this
					}
				]
			}]
		});

		//Call superclass constructor
		Zarafa.plugins.placecall.dialogs.PrepareCallContentPanel.superclass.constructor.call(this, config);
	},
	
	onOk : function()
	{
		var form = this.form.getForm();
		
		if(form.isValid()) {
			var values = form.getValues();
			
			container.getRequest().singleRequest('placecallmodule', 'placecall', {
				caller : values['caller'],
				callee : values['callee'],
				profile : 'placecall'
			}, new Zarafa.plugins.placecall.data.PlaceCallResponseHandler());
			this.close();
		}
	},
	
	onCancel : function()
	{
		this.close();
	}

});

//Register the dialog xtype
Ext.reg('placecall.preparecalldialog', Zarafa.plugins.placecall.dialogs.PrepareCallContentPanel);
