<?php
/*
 * Copyright 2012-2013 Holger de Carne
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License, version 3, 
 * as published by the Free Software Foundation with the following additional 
 * term according to sec. 7:
 *  
 * According to sec. 7 of the GNU Affero General Public License, version
 * 3, the terms of the AGPL are supplemented with the following terms:
 * 
 * "Zarafa" is a registered trademark of Zarafa B.V. The licensing of
 * the Program under the AGPL does not imply a trademark license.
 * Therefore any rights, title and interest in our trademarks remain
 * entirely with us.
 * 
 * However, if you propagate an unmodified version of the Program you are
 * allowed to use the term "Zarafa" to indicate that you distribute the
 * Program. Furthermore you may use our trademarks where it is necessary
 * to indicate the intended purpose of a product or service provided you
 * use it in accordance with honest practices in industrial or commercial
 * matters.  If you want to propagate modified versions of the Program
 * under the name "Zarafa" or "Zarafa Server", you may only do so if you
 * have a written permission by Zarafa B.V. (to acquire a permission
 * please contact Zarafa at trademark@zarafa.com).
 * 
 * The interactive user interface of the software displays an attribution
 * notice containing the term "Zarafa" and/or the logo of Zarafa.
 * Interactive user interfaces of unmodified and modified versions must
 * display Appropriate Legal Notices according to sec. 5 of the GNU
 * Affero General Public License, version 3, when you propagate
 * unmodified or modified versions of the Program. In accordance with
 * sec. 7 b) of the GNU Affero General Public License, version 3, these
 * Appropriate Legal Notices must retain the logo of Zarafa or display
 * the words "Initial Development by Zarafa" if the display of the logo
 * is not reasonably feasible for technical reasons. The use of the logo
 * of Zarafa in Legal Notices is allowed for unmodified and modified
 * versions of the software.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero General Public License for more details.
 *  
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

require_once('api/placecall_api.php');

/**
 * Placecall Plugin module
 *
 * This class processes the call invokations from the client and
 * triggers the PBX system accordingly.
 */
class PlacecallModule extends Module {

	/* Phone number keys for callee selection */
	private static $phoneNumberEntries = array(
			'home' => 'phone_number_home',
			'cellular' => 'phone_number_cellular',
			'assistant' => 'phone_number_assistant',
			'business' => 'phone_number_business',
			'business2' => 'phone_number_business2',
			'callback' => 'phone_number_callback',
			'car' => 'phone_number_car',
			'company' => 'phone_number_company',
			'home2' => 'phone_number_home2',
			'isdn' => 'phone_number_isdn',
			'other' => 'phone_number_other',
			'pager' => 'phone_number_pager',
			'pager' => 'phone_number_pager',
			'radio' => 'phone_number_radio',
			'telex' => 'phone_number_telex',
			'ttytdd' => 'phone_number_ttytdd'
			);
	
	/**
	 * Constructor
	 * @param int $id unique id.
	 * @param array $data list of all actions.
	 */
	public function PlacecallModule($id, $data) {
		parent::Module($id, $data);
	}

	/**
	 * Process the incoming events that were fired by the client.
	 * @return boolean True if everything was processed correctly.
	 */
	public function execute() {
		$result = false;
		foreach($this->data as $actionType => $actionData) {
			if(isset($actionType)) {
				$this->log("DEBUG: enter::{$actionType}");
				try {
					switch($actionType) {
						case 'preparefrominput':
							$result = $this->prepareFromInput($actionData);
							break;
						case 'preparefromrecord':
							$result = $this->prepareFromRecord($actionData);
							break;
						case 'placecall':
							$result = $this->placeCall($actionData);
							break;
						default:
							$this->handleUnknownActionType($actionType);
					}
				} catch (MAPIException $e) {
					$this->log("ERROR: Exception '{$e}'");
					$this->sendFeedback(false, $this->errorDetailsFromException($e));
				}
				$resultString = ($result ? 'true' : 'false');
				$this->log("DEBUG: exit::{$actionType} (result={$resultString})");
			}
		}
		return $result;
	}
	
	/**
	 * Prepare a direct call.
	 * @param mixed $action The data that holds the action request data.
	 * @return boolean True if the action succeeded.
	 **/
	private function prepareFromInput($action) {
		return $this->prepareResponse();
	}
	
	/**
	 * Prepare call based upon a Contact or AddressBook record.
	 * @param mixed $action The data that holds the action request data.
	 * @return boolean True if the action succeeded.
	 **/
	private function prepareFromRecord($action) {
		return $this->prepareResponse($action);
	}

	/**
	 * Prepare and send the response object to the client.
	 * @param mixed $action The AddressBook or Contact record containing the callee information or null if this is a direct call. 
	 * @return boolean True if action succeeded.
	 */
	private function prepareResponse($action = null) {
		$result = false;
		$response = array();
		$caller = $this->prepareCaller();
		if($caller !== false) {
			if(is_null($action)) {
				$response['callerReadOnly'] = PLUGIN_PLACECALL_CALLER_PROPERTY !== false;
				$response['caller'] = $caller;
				$response['calleesReadOnly'] = false;
				$response['callees'] = array('');
				$result = true;
			} else {
				$callees = $this->prepareCallees($action);
				if($callees !== false) {
					if(count($callees) > 0) {
						$response['callerReadOnly'] = PLUGIN_PLACECALL_CALLER_PROPERTY !== false;
						$response['caller'] = $caller;
						$response['calleesReadOnly'] = true;
						$response['callees'] = $callees;
						$result = true;
					} else {
						$response['message'] = 'Contact has no telephone numbers assigned.';
					}
				} else {
					$response['message'] = 'Unable to retrieve the contact information.';
				}
			}
		} else {
			$response['message'] = 'Unable to determine your caller id. Please contact your administrator.';
		}
		$this->addActionData(($result ? "prepare" : "error"), $response);
		$bus = $this->getBus();
		$bus->addData($this->getResponseData());
		return $result;
	}
	
	private function prepareCaller() {
		$caller = false;
		if(PLUGIN_PLACECALL_CALLER_PROPERTY) {
			$caller = $this->getUserCaller();
		} else {
			$settings = $this->getSettings();
			$caller = $settings->get('zarafa/v1/plugins/placecall/caller', '');
		}
		return $caller;
	}
	
	private function prepareCallees($action) {
		$callees = false;
		$callees = array();
		$this->log('DEBUG: Collecting callees for contact:');
		foreach(self::$phoneNumberEntries as $phoneNumberKey => $phoneNumberEntry) {
			$callee = false;
			if(isset($action[$phoneNumberEntry])) {
				$callee = trim($action[$phoneNumberEntry]);
			}
			if($callee != false) {
				$callees[$phoneNumberKey] = $this->normalize($callee);
				$this->log("DEBUG: Collecting phone number '{$phoneNumberKey}' = '{$callee}'");
			} else {
				$this->log("DEBUG: Skipping undefined phone number '{$phoneNumberEntry}'");
			}
		}
		return $callees;
	}
	
	/**
	 * Place a call for the selected numbers.
	 * @param mixed $action The data that holds the action request data.
	 * @return boolean True if the action succeeded.
	 **/
	private function placeCall($action) {
		$result = false;
		$response = array();
		$caller = $this->resolveActionCaller($action);
		$this->log("DEBUG: caller = '{$caller}'");
		if($caller !== false) {
			$callee = $this->resolveActionCallee($action);
			$this->log("DEBUG: callee = '{$callee}'");
			if($callee !== false) {
				$api = PlaceCallAPI::init($this, PLUGIN_PLACECALL_API);
				$result = $api->placeCall($caller, $callee);
				if($result !== false) {
					$response['caller'] = $caller;
					$response['callee'] = $callee;
				} else {
					$response['message'] = 'An error occured while placing the call. Please contact your administrator.';
				}
			} else {
				$response['message'] = 'Invalid \'Call number\' entry.';
			}
		} else {
			$response['message'] = 'Invalid \'Call via\' entry.';
		}
		$this->sendFeedback($result, $response);
		return $result;
	}
	
	private function resolveActionCaller($action) {
		$caller = false;
		if(isset($action['caller']) && !empty($action['caller'])) {
			if(PLUGIN_PLACECALL_CALLER_PROPERTY) {
				$caller = $this->getUserCaller();
			} else {
				$actionCaller = trim($action['caller']);
				if($actionCaller !== '') {
					$settings = $this->getSettings();
					$settings->set('zarafa/v1/plugins/placecall/caller', $actionCaller);
					$caller = $actionCaller;
				}
			}
		}
		return $caller;
	}
	
	private function resolveActionCallee($action) {
		$callee = false;
		if(isset($action['callee']) && !empty($action['callee'])) {
			$actionCallee = trim($action['callee']);
			if($actionCallee !== '') {
				$callee = $this->normalize($actionCallee);
			}
		}
		return $callee;
	}

	private function getUserCaller() {
		$caller = false;
		$session = $this->getSession();
		$user = $session->getUserName();
		$userEntryid = $session->getUserEntryID();
		$userEntryid || $this->log("ERROR: Unable to retrieve user entry id for user '{$user}'");
		$userEntry = mapi_ab_openentry($session->getAddressbook(), $userEntryid);
		$userEntry || $this->log("ERROR: Unable to retrieve user entry for user '{$user}'");
		$callerProps = mapi_getprops($userEntry, array(PLUGIN_PLACECALL_CALLER_PROPERTY));
		if($callerProps) {
			$callerProp = (isset($callerProps[PLUGIN_PLACECALL_CALLER_PROPERTY]) ? $callerProps[PLUGIN_PLACECALL_CALLER_PROPERTY] : '');
			if($callerProp != '') {
				$caller = $callerProp;
			} else {
				$this->log("ERROR: Caller property not set for user '{$user}'");
			}
		} else {
			$this->log("ERROR: Unable to retrieve user properties for user '{$user}'");
		}
		return $caller;
	}
	
	private function normalize($cid) {
		$cidN = preg_replace('/[^\+0-9]/', '', $cid);
		if(strpos($cidN, '+') === 0) {
			$cidN = PLUGIN_PLACECALL_INTPREFIX.substr($cidN, 1);
		} elseif(strpos($cidN, PLUGIN_PLACECALL_INTPREFIX) !== 0) {
			if(strpos($cidN, PLUGIN_PLACECALL_NATPREFIX) === 0) {
				$cidN = PLUGIN_PLACECALL_INTPREFIX.PLUGIN_PLACECALL_INTCODE.substr($cidN, strlen(PLUGIN_PLACECALL_NATPREFIX));
			} else {
				$cidN = PLUGIN_PLACECALL_INTPREFIX.PLUGIN_PLACECALL_INTCODE.PLUGIN_PLACECALL_NATCODE.$cidN;
			}
		}
		return $cidN;
	}
	
	public function log($message, $newline=true) {
		if(PLUGIN_PLACECALL_LOGGING_ENABLED) {
			$log = fopen(PLUGIN_PLACECALL_LOGGING_FILE, 'a');
			if($log) {
				fwrite($log, $message);
				if($newline) {
					fwrite($log, "\n");
				}
				fclose($log);
			}
		}
	}
	
	private function getSession() {
		return $GLOBALS['mapisession'];
	}

	private function getOperations() {
		return $GLOBALS["operations"];
	}
	
	private function getBus() {
		return $GLOBALS['bus'];
	}
	
	private function getSettings() {
		return $GLOBALS['settings'];
	}
	
}

?>
