<?
/**
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * PHP Google two-factor authentication module.
 *
 * See http://www.idontplaydarts.com/2011/07/google-totp-two-factor-authentication-for-php/
 * for more details
 *
 * @author Phil
 **/

define("LDAP_HOST", "127.0.0.1");
define("LDAP_BASE", "dc=directory,dc=local");
define("LDAP_USER", "cn=admin,dc=directory,dc=local");
define("LDAP_PASS", "password");

define("LDAP_USERNAME_RELATION_ATTRIBUTE", "mail");
define("LDAP_KEY_ATTRIBUTE", "carLicense");

define("LDAP_EXTRA_FILTER", "(!(objectClass=zarafa-contact))");

// Allow login if user has no key attribute
define("DEFAULT_ALLOW", TRUE);


class Google2FA {

	const keyRegeneration 	= 30;	// Interval between key regeneration
	const otpLength		= 6;	// Length of the Token generated

	private static $lut = array(	// Lookup needed for Base32 encoding
		"A" => 0,	"B" => 1,
		"C" => 2,	"D" => 3,
		"E" => 4,	"F" => 5,
		"G" => 6,	"H" => 7,
		"I" => 8,	"J" => 9,
		"K" => 10,	"L" => 11,
		"M" => 12,	"N" => 13,
		"O" => 14,	"P" => 15,
		"Q" => 16,	"R" => 17,
		"S" => 18,	"T" => 19,
		"U" => 20,	"V" => 21,
		"W" => 22,	"X" => 23,
		"Y" => 24,	"Z" => 25,
		"2" => 26,	"3" => 27,
		"4" => 28,	"5" => 29,
		"6" => 30,	"7" => 31
	);

	/**
	 * Generates a 16 digit secret key in base32 format
	 * @return string
	 **/
	public static function generate_secret_key($length = 16) {
		$b32 	= "234567QWERTYUIOPASDFGHJKLZXCVBNM";
		$s 	= "";

		for ($i = 0; $i < $length; $i++)
			$s .= $b32[rand(0,31)];

		return $s;
	}

	/**
	 * Returns the current Unix Timestamp devided by the keyRegeneration
	 * period.
	 * @return integer
	 **/
	public static function get_timestamp() {
		return floor(microtime(true)/self::keyRegeneration);
	}

	/**
	 * Decodes a base32 string into a binary string.
	 **/
	public static function base32_decode($b32) {

		$b32 	= strtoupper($b32);

		if (!preg_match('/^[ABCDEFGHIJKLMNOPQRSTUVWXYZ234567]+$/', $b32, $match))
			throw new Exception('Invalid characters in the base32 string.');

		$l 	= strlen($b32);
		$n	= 0;
		$j	= 0;
		$binary = "";

		for ($i = 0; $i < $l; $i++) {

			$n = $n << 5; 				// Move buffer left by 5 to make room
			$n = $n + self::$lut[$b32[$i]]; 	// Add value into buffer
			$j = $j + 5;				// Keep track of number of bits in buffer

			if ($j >= 8) {
				$j = $j - 8;
				$binary .= chr(($n & (0xFF << $j)) >> $j);
			}
		}

		return $binary;
	}

	/**
	 * Takes the secret key and the timestamp and returns the one time
	 * password.
	 *
	 * @param binary $key - Secret key in binary form.
	 * @param integer $counter - Timestamp as returned by get_timestamp.
	 * @return string
	 **/
	public static function oath_hotp($key, $counter)
	{
	    if (strlen($key) < 8)
		throw new Exception('Secret key is too short. Must be at least 16 base 32 characters');

	    $bin_counter = pack('N*', 0) . pack('N*', $counter);		// Counter must be 64-bit int
	    $hash 	 = hash_hmac ('sha1', $bin_counter, $key, true);

	    return str_pad(self::oath_truncate($hash), self::otpLength, '0', STR_PAD_LEFT);
	}

	/**
	 * Verifys a user inputted key against the current timestamp. Checks $window
	 * keys either side of the timestamp.
	 *
	 * @param string $b32seed
	 * @param string $key - User specified key
	 * @param integer $window
	 * @param boolean $useTimeStamp
	 * @return boolean
	 **/
	public static function verify_key($b32seed, $key, $window = 4, $useTimeStamp = true) {

		$timeStamp = self::get_timestamp();

		if ($useTimeStamp !== true) $timeStamp = (int)$useTimeStamp;

		$binarySeed = self::base32_decode($b32seed);

		for ($ts = $timeStamp - $window; $ts <= $timeStamp + $window; $ts++)
			if (self::oath_hotp($binarySeed, $ts) == $key)
				return true;

		return false;

	}

	/**
	 * Extracts the OTP from the SHA1 hash.
	 * @param binary $hash
	 * @return integer
	 **/
	public static function oath_truncate($hash)
	{
	    $offset = ord($hash[19]) & 0xf;

	    return (
	        ((ord($hash[$offset+0]) & 0x7f) << 24 ) |
	        ((ord($hash[$offset+1]) & 0xff) << 16 ) |
	        ((ord($hash[$offset+2]) & 0xff) << 8 ) |
	        (ord($hash[$offset+3]) & 0xff)
	    ) % pow(10, self::otpLength);
	}



}


function checktoken($token) {

	// If token is NOT present in POST then userkey needs to be fetched from ldap.
	if (! isset($_POST["token"])) {
		list ($ldap_result_key, $allow_login)=ldap_get_userkey();
	}

	// If userkey could be fetched, then set session variable, and show token login screen.
	if ($ldap_result_key != "") {
		$_SESSION["userkey"]=$ldap_result_key;
		//include("login.php");
		show_tokenpage();
		exit;
	}

	// If userkey session variable exists, then compare token.
	if (isset($_SESSION["userkey"])) {
  		$InitalizationKey = $_SESSION["userkey"];					// Set the inital key
		// Remove userkey from session variables, not needed anymore.
		$_SESSION["userkey"] = "";
  

  		$TimeStamp	  = Google2FA::get_timestamp();
  		$secretkey 	  = Google2FA::base32_decode($InitalizationKey);	// Decode it into binary
  		$otp       	  = Google2FA::oath_hotp($secretkey, $TimeStamp);	// Get current token

  		//echo("Init key: $InitalizationKey\n");
  		//echo("Timestamp: $TimeStamp\n");
  		//echo("One time password: $otp\n");


  		// Use this to verify a key as it allows for some time drift.
  		$result = Google2FA::verify_key($InitalizationKey, $token);
  		//var_dump($result);

  		if (!$result) {
    			$allow_login = FALSE;
  		}  else {
    			$allow_login = TRUE;
  		}


	}

	if ($allow_login) {
  		$_SESSION["tokenchecked"] = TRUE;
	} else {
		deny_login();
	}

}

function deny_login($logmsg="") {
        $_SESSION = array();
        $_SESSION["hresult"] = MAPI_E_LOGON_FAILED;
	if ($logmsg != "") {
        	error_log($logmsg);
	}
        header("Refresh: 0; url=index.php");
        exit;
}

function ldap_get_userkey() {
        // Connect to LDAP server and get the user DN
        $ldap = ldap_connect("ldap://".LDAP_HOST) or die('Could not connect to LDAP server.');
        ldap_set_option($ldap, LDAP_OPT_PROTOCOL_VERSION, 3);
        ldap_set_option($ldap, LDAP_OPT_REFERRALS, 0);
        @ldap_bind($ldap, LDAP_USER, LDAP_PASS) or die('Could not bind to LDAP.');

        $ldap_query = "(&".LDAP_EXTRA_FILTER."(".LDAP_USERNAME_RELATION_ATTRIBUTE."=".$_SESSION["username"]."))";
        $ldap_attrs = array("dn");
        $ldap_result = ldap_search($ldap, LDAP_BASE, $ldap_query, $ldap_attrs);

        $ldap_entries = ldap_get_entries($ldap, $ldap_result);

        $allow_login = FALSE;

        if ($ldap_entries["count"] == "0") {
                // 0 Results
                deny_login("0 Results while checking token for: ".$_SESSION["username"]);
        } elseif ($ldap_entries["count"] > 1) {
                // Ambigous result
                deny_login("Ambigous Result while checking token for: ".$_SESSION["username"]);
        } else {
                // 1 result: OK
                //echo print_r($ldap_entries,1) . "<BR>";
                //echo "Keycount: ". $ldap_entries[0][$LDAP_KEY_ATTRIBUTE]["count"];
                $ldap_result_dn = $ldap_entries[0]["dn"];
        }

        ldap_unbind($ldap);

        // Bind to ldap with webaccess username and get key
        $LDAP_KEY_ATTRIBUTE = strtolower(LDAP_KEY_ATTRIBUTE);

        $ldap = ldap_connect("ldap://".LDAP_HOST) or die('Could not connect to LDAP server.');
        ldap_set_option($ldap, LDAP_OPT_PROTOCOL_VERSION, 3);
        ldap_set_option($ldap, LDAP_OPT_REFERRALS, 0);
        @ldap_bind($ldap, $ldap_result_dn, $_POST["password"]) or deny_login("Could not bind to ldap server for user: ".$ldap_result_dn);

        $ldap_attrs = array(LDAP_KEY_ATTRIBUTE);
        $ldap_result = ldap_read($ldap, $ldap_result_dn, "(objectClass=*)", $ldap_attrs);

        $ldap_entries = ldap_get_entries($ldap, $ldap_result);

        if ($ldap_entries[0][$LDAP_KEY_ATTRIBUTE]["count"] == 0) {
                // User key not found
                $allow_login = DEFAULT_ALLOW;
        } else {
                $ldap_result_key = $ldap_entries[0][$LDAP_KEY_ATTRIBUTE][0];
        }

        ldap_unbind($ldap);

	return array($ldap_result_key, $allow_login);
}

function show_tokenpage() {
	$user = htmlentities($_GET["user"]);
	header("Content-type: text/html; charset=utf-8");
?><!DOCTYPE HTML PUBLIC "-//W3C//DTD HTML 4.01 Transitional//EN" "http://www.w3.org/TR/html4/loose.dtd">
<html>
	<head>
		<title>Zarafa WebAccess</title>
		<link rel="stylesheet" type="text/css" href="client/layout/css/login.css">
		<link rel="icon" href="client/layout/img/favicon.ico"  type="image/x-icon">
		<link rel="shortcut icon" href="client/layout/img/favicon.ico" type="image/x-icon">	
		<script type="text/javascript">
			window.onload = function(){
					document.getElementById("token").focus();
			}
		</script>
	</head>
	<body class="login">
		<table id="layout">
			<tr><td>
				<div id="login_main">
					<form action="index.php?logon<?=($user)?'&user='.$user:''?>" method="post">
					<!-- Store action attributes to hidden variable to pass it to index page -->
					<?php	if($_POST && $_POST["action_url"] != "") {	?>
								<!-- if login has failed then action attributes will be in POST variable -->
								<input type="hidden" name="action_url" value="<?= htmlspecialchars($_POST["action_url"]) ?>"></input>
					<?php	} else {	?>
								<!-- or else in the URL -->
								<input type="hidden" name="action_url" value="<?= stristr($_SERVER["REQUEST_URI"], "?action=") ?>"></input>
					<?php	}	?>
					
						<div id="login_data">
							<p>Token needed for this user.</p>
							<p class="error"><?php

	if (isset($_SESSION) && isset($_SESSION["hresult"])) {
		switch($_SESSION["hresult"]){
			case MAPI_E_LOGON_FAILED:
			case MAPI_E_UNCONFIGURED:
				echo _("Logon failed, please check your name/password.");
				break;
			case MAPI_E_NETWORK_ERROR:
				echo _("Cannot connect to the Zarafa Server.");
				break;
			default:
				echo "Unknown MAPI Error: ".get_mapi_error_name($_SESSION["hresult"]);
		}
		unset($_SESSION["hresult"]);
	}else if (isset($_GET["logout"]) && $_GET["logout"]=="auto"){
		echo _("You have been automatically logged out");
	}else{
		echo "&nbsp;";
	}
							?></p>
							<table id="form_fields">
								<tr>
									<th>Token:</th>
									<td><input type="token" name="token" id="token" class="inputelement"></td>
								</tr>
								<tr>
									<td>&nbsp;</td>
								</tr>
								<tr>
									<td>&nbsp;</td>
									<td><input id="submitbutton" type="submit" value=<?=_("Logon")?>></td>
								</tr>
							</table>
						</div>
					</form>
					<span id="version"><?=defined("DEBUG_SERVER_ADDRESS")?"Server: ".DEBUG_SERVER_ADDRESS." - ":""?><?=phpversion("mapi")?><?=defined("SVN")?"-svn".SVN:""?></span>
				</div>
			</td></tr>
		</table>
	</body>
</html>

<?php

}
