Ext.namespace('Zarafa.plugins.owncloud');

/**
 * @class Zarafa.plugins.owncloud.ABOUT
 * @extends String
 *
 * The copyright string holding the copyright notice for the Zarafa owncloud Plugin.
 */
Zarafa.plugins.owncloud.ABOUT = ""
	+ "<p>Copyright (C) 2012-2013  Christoph Haas &lt;christoph.@sprinternet.at&gt;</p>"

	+ "<p>This program is free software; you can redistribute it and/or "
	+ "modify it under the terms of the GNU Lesser General Public "
	+ "License as published by the Free Software Foundation; either "
	+ "version 2.1 of the License, or (at your option) any later version.</p>"

	+ "<p>This program is distributed in the hope that it will be useful, "
	+ "but WITHOUT ANY WARRANTY; without even the implied warranty of "
	+ "MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU "
	+ "Lesser General Public License for more details.</p>"

	+ "<p>You should have received a copy of the GNU Lesser General Public "
	+ "License along with this program; if not, write to the Free Software "
	+ "Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA</p>"

	+ "<hr />"

	+ "<p>The owncloud plugin contains the following third-party components:</p>"

	+ "<h1>WebDAV Client API</h1>"

	+ "<p>Copyright (C) 2003 Christian Juerges &lt;christian.juerges@xwave.ch&gt;</p>"

	+ "<p>Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation files (the \"Software\"), to deal in the Software without restriction, including without limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is furnished to do so, subject to the following conditions:</p>"

	+ "<p>The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.</p>"

	+ "<p>THE SOFTWARE IS PROVIDED \"AS IS\", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.</p>";
Ext.namespace('Zarafa.plugins.owncloud.data');

/**
 * @class Zarafa.plugins.owncloud.data.ResponseHandler
 * @extends Zarafa.core.data.AbstractResponseHandler
 *
 * Owncloud specific response handler.
 */
Zarafa.plugins.owncloud.data.ResponseHandler = Ext.extend(Zarafa.core.data.AbstractResponseHandler, {

	/**
	 * @cgf {String} The id of the opened node in fuile tree recieved from the Owncloud
	 */
	nodeId: undefined,

	/**
	 * @cfg {Function} successCallback The function which
	 * will be called after success request.
	 */
	successCallback : null,

	/**
	 * Call the successCallback callback function.
	 * @param {Object} response Object contained the response data.
	 */
	doLoaddirectory : function(response)
	{
		this.successCallback(response.items, response);
	},
	
	/**
	 * Call the successCallback callback function.
	 * @param {Object} response Object contained the response data.
	 */
	doGetversion : function(response)
	{
		this.successCallback(response);
	},
	
	/**
	 * Call the successCallback callback function.
	 * @param {Object} response Object contained the response data.
	 */	
	doTmpdownload : function(response)
	{
		this.successCallback(response.items, response);
	},
	
	/**
	 * Call the successCallback callback function.
	 * @param {Object} response Object contained the response data.
	 */
	doPrepareattachment : function(response)
	{
		this.successCallback(response);
	},
	
	/**
	 * Call the successCallback callback function.
	 * @param {Object} response Object contained the response data.
	 */
	doCreatedir : function(response)
	{
		this.successCallback(response);
	},
	
	/**
	 * Call the successCallback callback function.
	 * @param {Object} response Object contained the response data.
	 */
	doUploadtooc : function(response)
	{
		this.successCallback(response);
	},

	/**
	 * In case exception happened on server, server will return
	 * exception response with the code of exception.
	 * @param {Object} response Object contained the response data.
	 */
	doError: function(response)
	{
		alert("error response code: " + response.error.info.code);
	}
});

Ext.reg('zarafa.owncloudresponsehandler', Zarafa.plugins.owncloud.data.ResponseHandler);Ext.namespace('Zarafa.plugins.owncloud.data');

/**
 * @class Zarafa.plugins.owncloud.data.DirectoryLoader
 * @extends Ext.tree.TreeLoader
 *
 * Owncloud Directory loader. Extends Ext treeloader to use Zarafa
 * specific requests.
 */
Zarafa.plugins.owncloud.data.DirectoryLoader = Ext.extend(Ext.tree.TreeLoader, {

	/**
	 * also load files
	 */
	files : false,
	
	/**
	 * @constructor
	 * @param {Object} config Configuration object
	 */
	constructor : function(config)
	{
		config = config || {};
		
		if(undefined !== config.loadfiles) {
			this.files = config.loadfiles;
		}

		Ext.applyIf(config, {
			preloadChildren: true,
			directFn : this.loadFolder.createDelegate(this)
		});

		Zarafa.plugins.owncloud.data.DirectoryLoader.superclass.constructor.call(this, config);
	},

	/**
	 * Will do single request to owncloud module with provided nodeId and
	 * in case of success will load the content of this node.
	 *
	 * @param {Number} nodeId The id of node which content need to be loaded
	 * @param {Function} callback The function which need to be called after response received
	 */
	loadFolder : function(nodeId, callback)
	{
		var self = this;
		var responseHandler = new Zarafa.plugins.owncloud.data.ResponseHandler({
			successCallback: callback,
			nodeId: nodeId
		});
		
		container.getRequest().singleRequest(
			'owncloudmodule',
			'loaddirectory',
			{
				id : nodeId,
				loadfiles : this.files
			},
			responseHandler
		);
	}
});Ext.namespace('Zarafa.plugins.owncloud.data');

/**
 * @class Zarafa.plugins.owncloud.data.Version
 * @extends Object
 *
 * Global Version Manager
 */
Zarafa.plugins.owncloud.data.Version = Ext.extend(Object, {
	ocversion : undefined,
	plversion : undefined,
	
	dTask: undefined,
	
	init : function() {
	
		this.dTask = new Ext.util.DelayedTask(function(){
			var responseHandler = new Zarafa.plugins.owncloud.data.ResponseHandler({
				successCallback: this.gotVersion.createDelegate(this)
			});
			container.getRequest().singleRequest(
				'owncloudmodule',
				'getversion',
				{
					plugin : "owncloud"
				},
				responseHandler
			);
		}, this);
		this.dTask.delay(5000);	// delay status detection to improve webapp load time...
	},
	
	gotVersion : function(response){
		this.ocversion = response.ocversion;
		this.plversion = response.version;
	},
	
	getPluginVersion : function() {
		return this.plversion;
	},
	
	getOwncloudVersion : function() {
		return this.ocversion;
	}	
});

// Make it a singleton
Zarafa.plugins.owncloud.data.Version = new Zarafa.plugins.owncloud.data.Version();Ext.namespace('Zarafa.plugins.owncloud.settings');

/**
 * @class Zarafa.plugins.owncloud.settings.SettingsOwncloudComponentsWidget
 * @extends Zarafa.settings.ui.SettingsWidget
 * @xtype Zarafa.plugins.owncloud.settingsowncloudwidget
 *
 * The {@link Zarafa.settings.ui.SettingsWidget widget} for configuring
 * the general owncloud options in the {@link Zarafa.owncloud.settings.SettingsOwncloudComponentsCategory owncloud category}.
 */
Zarafa.plugins.owncloud.settings.SettingsOwncloudComponentsWidget = Ext.extend(Zarafa.settings.ui.SettingsWidget, {

	/**
	 * @constructor
	 * @param {Object} config Configuration object
	 */
	constructor : function(config)
	{
		config = config || {};

		Ext.applyIf(config, {
			title : _('Enable/Disable owncloud components'),
			layout : 'form',
			items : [{
				xtype : 'checkbox',
				name : 'zarafa/v1/plugins/owncloud/enable_attach_from_oc',
				ref : 'attachFrom',
				fieldLabel : _('Attach a file from owncloud'),
				lazyInit : false,
				listeners : {
					check : this.onCheck,
					scope : this
				}
			},{
				xtype : 'checkbox',
				name : 'zarafa/v1/plugins/owncloud/enable_store_to_oc',
				ref : 'attachTo',
				fieldLabel : _('Save received attachments to owncloud'),
				lazyInit : false,
				listeners : {
					check : this.onCheck,
					scope : this
				}
			},{
				xtype : 'checkbox',
				name : 'zarafa/v1/plugins/owncloud/enable_browser',
				ref : 'enableBrowser',
				fieldLabel : _('Enable WebDav Browser (EXPERIMENTAL!!)'),
				lazyInit : false,
				listeners : {
					check : this.onCheck,
					scope : this
				}
			}]
		});

		Zarafa.plugins.owncloud.settings.SettingsOwncloudComponentsWidget.superclass.constructor.call(this, config);
	},

	/**
	 * Called by the {@link Zarafa.settings.ui.SettingsCategory Category} when
	 * it has been called with {@link zarafa.settings.ui.SettingsCategory#update}.
	 * This is used to load the latest version of the settings from the
	 * {@link Zarafa.settings.SettingsModel} into the UI of this category.
	 * @param {Zarafa.settings.SettingsModel} settingsModel The settings to load
	 */
	update : function(settingsModel)
	{
		this.model = settingsModel;
		this.attachFrom.setValue(settingsModel.get(this.attachFrom.name));
		this.attachTo.setValue(settingsModel.get(this.attachTo.name));
		this.enableBrowser.setValue(settingsModel.get(this.enableBrowser.name));
	},

	/**
	 * Called by the {@link Zarafa.settings.ui.SettingsCategory Category} when
	 * it has been called with {@link zarafa.settings.ui.SettingsCategory#updateSettings}.
	 * This is used to update the settings from the UI into the {@link Zarafa.settings.SettingsModel settings model}.
	 * @param {Zarafa.settings.SettingsModel} settingsModel The settings to update
	 */
	updateSettings : function(settingsModel)
	{
		settingsModel.set(this.attachFrom.name, this.attachFrom.getValue());
		settingsModel.set(this.attachTo.name, this.attachTo.getValue());
		settingsModel.set(this.enableBrowser.name, this.enableBrowser.getValue());
	},

	/**
	 * Event handler called when checkbox has been modified
	 *
	 * @param {Ext.form.CheckBox} checkbox Checkbox element from which the event originated
	 * @param {Boolean} checked State of the checkbox
	 * @private
	 */
	onCheck : function(checkbox, checked)
	{
		if(this.model) {
			// FIXME: The settings model should be able to detect if
			// a change was applied
			if (this.model.get(checkbox.name) !== checked) {
				this.model.set(checkbox.name, checked);
			}
		}
	}
});

Ext.reg('Zarafa.plugins.owncloud.settingsowncloudcomponentswidget', Zarafa.plugins.owncloud.settings.SettingsOwncloudComponentsWidget);
Ext.namespace('Zarafa.plugins.owncloud.settings');

/**
 * @class Zarafa.owncloud.settings.SettingsOwncloudWidget
 * @extends Zarafa.settings.ui.SettingsWidget
 * @xtype Zarafa.plugins.owncloud.settingsowncloudwidget
 *
 * The {@link Zarafa.settings.ui.SettingsWidget widget} for configuring
 * the general owncloud options in the {@link Zarafa.owncloud.settings.SettingsOwncloudCategory owncloud category}.
 */
Zarafa.plugins.owncloud.settings.SettingsOwncloudWidget = Ext.extend(Zarafa.settings.ui.SettingsWidget, {

	Base64 : {
		// private property
		_keyStr : "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/=",

		// public method for encoding
		encode : function (input) {
			var output = "";
			var chr1, chr2, chr3, enc1, enc2, enc3, enc4;
			var i = 0;

			input = this._utf8_encode(input);

			while (i < input.length) {

				chr1 = input.charCodeAt(i++);
				chr2 = input.charCodeAt(i++);
				chr3 = input.charCodeAt(i++);

				enc1 = chr1 >> 2;
				enc2 = ((chr1 & 3) << 4) | (chr2 >> 4);
				enc3 = ((chr2 & 15) << 2) | (chr3 >> 6);
				enc4 = chr3 & 63;

				if (isNaN(chr2)) {
					enc3 = enc4 = 64;
				} else if (isNaN(chr3)) {
					enc4 = 64;
				}

				output = output +
				this._keyStr.charAt(enc1) + this._keyStr.charAt(enc2) +
				this._keyStr.charAt(enc3) + this._keyStr.charAt(enc4);

			}

			return output;
		},
		
		// public method for decoding
		decode : function (input) {
			var output = "";
			var chr1, chr2, chr3;
			var enc1, enc2, enc3, enc4;
			var i = 0;

			input = input.replace(/[^A-Za-z0-9\+\/\=]/g, "");

			while (i < input.length) {

				enc1 = this._keyStr.indexOf(input.charAt(i++));
				enc2 = this._keyStr.indexOf(input.charAt(i++));
				enc3 = this._keyStr.indexOf(input.charAt(i++));
				enc4 = this._keyStr.indexOf(input.charAt(i++));

				chr1 = (enc1 << 2) | (enc2 >> 4);
				chr2 = ((enc2 & 15) << 4) | (enc3 >> 2);
				chr3 = ((enc3 & 3) << 6) | enc4;

				output = output + String.fromCharCode(chr1);

				if (enc3 != 64) {
					output = output + String.fromCharCode(chr2);
				}
				if (enc4 != 64) {
					output = output + String.fromCharCode(chr3);
				}

			}

			output = this._utf8_decode(output);

			return output;

		},
		
		// private method for UTF-8 decoding
		_utf8_decode : function (utftext) {
			var string = "";
			var i = 0;
			var c = 0
			var c1 = 0;
			var c2 = 0;
			var c3 = 0;

			while ( i < utftext.length ) {

				c = utftext.charCodeAt(i);

				if (c < 128) {
					string += String.fromCharCode(c);
					i++;
				}
				else if((c > 191) && (c < 224)) {
					c2 = utftext.charCodeAt(i+1);
					string += String.fromCharCode(((c & 31) << 6) | (c2 & 63));
					i += 2;
				}
				else {
					c2 = utftext.charCodeAt(i+1);
					c3 = utftext.charCodeAt(i+2);
					string += String.fromCharCode(((c & 15) << 12) | ((c2 & 63) << 6) | (c3 & 63));
					i += 3;
				}

			}

			return string;
		},

		// private method for UTF-8 encoding
		_utf8_encode : function (string) {
			string = string.replace(/\r\n/g,"\n");
			var utftext = "";

			for (var n = 0; n < string.length; n++) {

				var c = string.charCodeAt(n);

				if (c < 128) {
					utftext += String.fromCharCode(c);
				}
				else if((c > 127) && (c < 2048)) {
					utftext += String.fromCharCode((c >> 6) | 192);
					utftext += String.fromCharCode((c & 63) | 128);
				}
				else {
					utftext += String.fromCharCode((c >> 12) | 224);
					utftext += String.fromCharCode(((c >> 6) & 63) | 128);
					utftext += String.fromCharCode((c & 63) | 128);
				}

			}

			return utftext;
		}
	},

	/**
	 * @constructor
	 * @param {Object} config Configuration object
	 */
	constructor : function(config)
	{
		config = config || {};

		Ext.applyIf(config, {
			title : _('General owncloud settings'),
			layout : 'form',
			items : [{
				xtype : 'textfield',
				name : 'zarafa/v1/contexts/owncloud/server',
				ref : 'serverField',
				fieldLabel : _('Server Address'),
				emptyText : _('demo.owncloud.org'),
				anchor : '100%',
				listeners : {
					change : this.onFieldChange,
					scope : this
				}
			},{
				xtype : 'textfield',
				name : 'zarafa/v1/contexts/owncloud/port',
				ref : 'portField',
				fieldLabel : _('Server Port'),
				emptyText : '80',
				anchor : '100%',
				listeners : {
					change : this.onFieldChange,
					scope : this
				}
			},{
				xtype : 'textfield',
				name : 'zarafa/v1/contexts/owncloud/port_ssl',
				ref : 'portsslField',
				fieldLabel : _('Server SSL Port'),
				emptyText : '443',
				anchor : '100%',
				listeners : {
					change : this.onFieldChange,
					scope : this
				}
			},{
				xtype : 'textfield',
				name : 'zarafa/v1/contexts/owncloud/owncloud_path',
				ref : 'pathField',
				fieldLabel : _('Path to Owncloud/Webdav'),
				emptyText : _('/files/webdav.php'),
				anchor : '100%',
				listeners : {
					change : this.onFieldChange,
					scope : this
				}
			},{
				xtype : 'textfield',
				name : 'zarafa/v1/contexts/owncloud/username',
				ref : 'usernameField',
				fieldLabel : _('Username'),
				emptyText : _('test'),
				anchor : '100%',
				listeners : {
					change : this.onFieldChange,
					scope : this
				}
			},{
				xtype : 'textfield',
				name : 'zarafa/v1/contexts/owncloud/password',
				ref : 'passwordField',
				inputType : 'password',
				fieldLabel : _('Password'),
				emptyText : _('test'),
				anchor : '100%',
				listeners : {
					change : this.onFieldChange,
					scope : this
				}
			},{
				xtype : 'checkbox',
				name : 'zarafa/v1/contexts/owncloud/use_ssl',
				ref : 'useSSL',
				fieldLabel : _('Use SSL connections'),
				lazyInit : false,
				listeners : {
					check : this.onCheck,
					scope : this
				}
			}]
		});

		Zarafa.plugins.owncloud.settings.SettingsOwncloudWidget.superclass.constructor.call(this, config);
	},

	/**
	 * Called by the {@link Zarafa.settings.ui.SettingsCategory Category} when
	 * it has been called with {@link zarafa.settings.ui.SettingsCategory#update}.
	 * This is used to load the latest version of the settings from the
	 * {@link Zarafa.settings.SettingsModel} into the UI of this category.
	 * @param {Zarafa.settings.SettingsModel} settingsModel The settings to load
	 */
	update : function(settingsModel)
	{
		this.model = settingsModel;
		this.serverField.setValue(settingsModel.get(this.serverField.name));
		this.portField.setValue(settingsModel.get(this.portField.name));
		this.portsslField.setValue(settingsModel.get(this.portsslField.name));
		this.pathField.setValue(settingsModel.get(this.pathField.name));
		this.usernameField.setValue(settingsModel.get(this.usernameField.name));
		this.passwordField.setValue(this.Base64.decode(settingsModel.get(this.passwordField.name)));
		this.useSSL.setValue(settingsModel.get(this.useSSL.name));
		
		if (settingsModel.get(this.useSSL.name) == true) {
			this.portsslField.show();
			this.portsslField.label.show();
			this.portField.hide();
			this.portField.label.hide();
		} else {
			this.portsslField.hide();
			this.portsslField.label.hide();
			this.portField.show();
			this.portField.label.show();
		}
	},

	/**
	 * Called by the {@link Zarafa.settings.ui.SettingsCategory Category} when
	 * it has been called with {@link zarafa.settings.ui.SettingsCategory#updateSettings}.
	 * This is used to update the settings from the UI into the {@link Zarafa.settings.SettingsModel settings model}.
	 * @param {Zarafa.settings.SettingsModel} settingsModel The settings to update
	 */
	updateSettings : function(settingsModel)
	{
		// We must either set the requested subject, or the default subject
		var server 	 = this.serverField.getValue() 	 || this.serverField.emptyText;
		var port 	 = this.portField.getValue() 	 || this.portField.emptyText;
		var portssl  = this.portsslField.getValue()  || this.portsslField.emptyText;
		var path 	 = this.pathField.getValue() 	 || this.pathField.emptyText;
		var username = this.usernameField.getValue() || this.usernameField.emptyText;
		var password = this.passwordField.getValue() || this.passwordField.emptyText;
		
		settingsModel.beginEdit();
		settingsModel.set(this.serverField.name, server);
		settingsModel.set(this.portField.name, port);
		settingsModel.set(this.portsslField.name, portssl);
		settingsModel.set(this.pathField.name, path);
		settingsModel.set(this.usernameField.name, username);
		settingsModel.set(this.passwordField.name, this.Base64.encode(password));
		settingsModel.set(this.useSSL.name, this.useSSL.getValue());
		settingsModel.endEdit();
	},

	/**
	 * Event handler called when checkbox has been modified
	 *
	 * @param {Ext.form.CheckBox} checkbox Checkbox element from which the event originated
	 * @param {Boolean} checked State of the checkbox
	 * @private
	 */
	onCheck : function(checkbox, checked)
	{
		if(this.model) {
			// FIXME: The settings model should be able to detect if
			// a change was applied
			if (this.model.get(checkbox.name) !== checked) {
				this.model.set(checkbox.name, checked);
			}
		}
		
		if (checked) {
			this.portsslField.show();
			this.portsslField.label.show();
			this.portField.hide();
			this.portField.label.hide();
		} else {
			this.portsslField.hide();
			this.portsslField.label.hide();
			this.portField.show();
			this.portField.label.show();
		}
	},
	
	/**
	 * Event handler which is called when one of the textfields has been changed.
	 * This will apply the new value to the settings (or if the text is empty,
	 * the {@link Ext.form.Field#emptyText} will be saved).
	 * @param {Ext.form.Field} field The field which has fired the event
	 * @param {String} value The new value
	 * @private
	 */
	onFieldChange : function(field, value)
	{
		if (this.model) {
			this.model.set(field.name, value || field.emptyText);
		}
	}
});

Ext.reg('Zarafa.plugins.owncloud.settingsowncloudwidget', Zarafa.plugins.owncloud.settings.SettingsOwncloudWidget);
Ext.namespace('Zarafa.plugins.owncloud.settings');

/**
 * @class Zarafa.owncloud.settings.SettingsOwncloudVersionWidget
 * @extends Zarafa.settings.ui.SettingsWidget
 * @xtype Zarafa.plugins.owncloud.settingsowncloudwidget
 *
 * The {@link Zarafa.settings.ui.SettingsWidget widget} for configuring
 * the general owncloud options in the {@link Zarafa.owncloud.settings.SettingsOwncloudCategory owncloud category}.
 */
Zarafa.plugins.owncloud.settings.SettingsOwncloudVersionWidget = Ext.extend(Zarafa.settings.ui.SettingsWidget, {
	/**
	 * @constructor
	 * @param {Object} config Configuration object
	 */
	constructor : function(config)
	{
		config = config || {};
		
		Ext.applyIf(config, {
			title : _('Version Info'),
			layout : 'form',
			items :[{
				xtype : 'displayfield',
				fieldLabel : _('Owncloud'),
				value : Zarafa.plugins.owncloud.data.Version.getOwncloudVersion(),
				htmlEncode : true
			},{
				xtype : 'displayfield',
				fieldLabel : _('Plugin'),
				value : Zarafa.plugins.owncloud.data.Version.getPluginVersion(),
				htmlEncode : true
			}]
		});

		Zarafa.plugins.owncloud.settings.SettingsOwncloudVersionWidget.superclass.constructor.call(this, config);
	}
});

Ext.reg('Zarafa.plugins.owncloud.settingsowncloudversionwidget', Zarafa.plugins.owncloud.settings.SettingsOwncloudVersionWidget);
Ext.namespace('Zarafa.plugins.owncloud.settings');

/**
 * @class Zarafa.plugins.owncloud.settings.SettingsOwncloudCategory
 * @extends Zarafa.settings.ui.SettingsCategory
 * @xtype Zarafa.plugins.owncloud.settingsowncloudcategory
 *
 * The owncloud category for users which will
 * allow the user to configure Owncloud related settings
 */
Zarafa.plugins.owncloud.settings.SettingsOwncloudCategory = Ext.extend(Zarafa.settings.ui.SettingsCategory, {
	// Insertion points for this class
	/**
	 * @insert context.settings.category.owncloud
	 * Insertion point to register new {@link Zarafa.settings.ui.SettingsWidget widgets}
	 * for the {@link Zarafa.owncloud.settings.SettingsOwncloudCategory Owncoud Category}.
	 * @param {Zarafa.owncloud.settings.SettingsOwncloudCategory} category The owncloud
	 * category to which the widgets will be added.
	 */

	/**
	 * @constructor
	 * @param {Object} config Configuration object
	 */
	constructor : function(config)
	{
		config = config || {};

		Ext.applyIf(config, {
			title : container.getSettingsModel().get('zarafa/v1/plugins/owncloud/button_name'),
			categoryIndex : 1,
			iconCls : 'icon_owncloud_category',
			items : [{
				xtype : 'Zarafa.plugins.owncloud.settingsowncloudcomponentswidget'	// enable/disable plugins
			},{
				xtype : 'Zarafa.plugins.owncloud.settingsowncloudwidget'				// owncloud specific settings like server address...
			},{
				xtype : 'Zarafa.plugins.owncloud.settingsowncloudversionwidget'				// owncloud plugin version
			},
			container.populateInsertionPoint('context.settings.category.owncloud', this)
			]
		});

		Zarafa.plugins.owncloud.settings.SettingsOwncloudCategory.superclass.constructor.call(this, config);
	}
});

Ext.reg('Zarafa.plugins.owncloud.settingsowncloudcategory', Zarafa.plugins.owncloud.settings.SettingsOwncloudCategory);
Ext.namespace('Zarafa.plugins.owncloud.ui');

/**
 * @class Zarafa.plugins.owncloud.ui.AttachFromOwncloudTreePanel
 * @extends Ext.tree.TreePanel
 * Shows tree of all user files from Owncloud
 * @xtype Zarafa.plugins.owncloud.attachfromowncloudtreepanel
 */
Zarafa.plugins.owncloud.ui.AttachFromOwncloudTreePanel = Ext.extend(Ext.tree.TreePanel, {

	/**
	 * @constructor
	 * @param {Object} config
	 */
	constructor : function(config)
	{
		config = config || {};
		Ext.applyIf(config, {
			root: {
				nodeType: 'async',
				text: '/',
				id: '/',
				expanded : true
			},
			autoScroll: true,
			viewConfig: {
				style	:	{ overflow: 'auto', overflowX: 'hidden' }
			},
			maskDisabled: true,
			loader : new Zarafa.plugins.owncloud.data.DirectoryLoader({loadfiles: true}),
			buttons : [
				this.createActionButtons()
			]
		});
		Zarafa.plugins.owncloud.ui.AttachFromOwncloudTreePanel.superclass.constructor.call(this, config);
	},

	/**
	 * Creates action button for adding file
	 * from Owncloud to email as attachment
	 * @return {Array}
	 * @private
	 */
	createActionButtons : function()
	{
		return [{
			xtype   : 'button',
			text    : _('&nbsp;&nbsp;Add Attachment'),
			tooltip : {
				title   : _('Add Attachment'),
				text    : _('Add selected attachment from owncloud to email attachment.')
			},
			width	: 120,
			iconCls : 'icon_owncloud',
			handler : this.downloadSelectedFilesFromOwncloudToTmp,
			scope   : this
		}];
	},

	/**
	 * Sends a query with ids of selected files from user's Owncloud folder
	 * @private
	 */
	downloadSelectedFilesFromOwncloudToTmp : function()
	{
		var selectedNodes =  this.dialog.treePanel.getChecked();
		var idsList = [];
		var emailRecord = this.dialog.record;
		var attachmentStore = emailRecord.getAttachmentStore();

		Ext.each(selectedNodes, function(node, index) {
			idsList.push( node.id );
		});
		
		if(idsList.length < 1) {
			Zarafa.common.dialogs.MessageBox.show({
				title   : _('Warning'),
				msg     : _('You have to choose at least one file!'),
				icon    : Zarafa.common.dialogs.MessageBox.WARNING,
				buttons : Zarafa.common.dialogs.MessageBox.OK
			});
		} else {
			try {
				this.dialog.treePanel.disable();
				container.getRequest().singleRequest(
					'owncloudmodule',
					'download-to-tmp',
					{
						ids : idsList,
						dialog_attachments: attachmentStore.getId()
					},
					new Zarafa.plugins.owncloud.data.ResponseHandler({
						successCallback : this.addDownloadedFilesAsAttachmentToEmail.createDelegate(this)
					})
				);
			} catch (e) {
				Zarafa.common.dialogs.MessageBox.show({
					title   : _('Warning'),
					msg     : e.getMessage(),
					icon    : Zarafa.common.dialogs.MessageBox.WARNING,
					buttons : Zarafa.common.dialogs.MessageBox.OK
				});
			}
		}
	},

	/**
	 * Converts received file information to attachment record
	 * @param {Object} downloadedFileInfo
	 * @private
	 */
	convertDownloadedFileInfoToAttachmentRecord : function(downloadedFileInfo)
	{
		var attachmentRecord = Zarafa.core.data.RecordFactory.createRecordObjectByObjectType(Zarafa.core.mapi.ObjectType.MAPI_ATTACH);

		attachmentRecord.set('tmpname', downloadedFileInfo.tmpname);
		attachmentRecord.set('name', downloadedFileInfo.name);
		attachmentRecord.set('size', downloadedFileInfo.size);
		return attachmentRecord;
	},

	/**
	 * Adds files received from the user's Owncloud folder to
	 * the email as attachments
	 * @param {Object} downloadedFilesInfoArray
	 * @private
	 */
	addDownloadedFilesAsAttachmentToEmail : function(downloadedFilesInfoArray)
	{
		var emailRecord = this.dialog.record;
		var attachmentStore = emailRecord.getAttachmentStore();

		var attachmentRecord = null;
		Ext.each(downloadedFilesInfoArray, function(downloadedFileInfo, index) {
			attachmentRecord = this.convertDownloadedFileInfoToAttachmentRecord(downloadedFileInfo);
			attachmentStore.add(attachmentRecord);
		}, this);
		this.dialog.close();
	}
});

Ext.reg('Zarafa.plugins.owncloud.attachfromowncloudtreepanel', Zarafa.plugins.owncloud.ui.AttachFromOwncloudTreePanel);
Ext.namespace('Zarafa.plugins.owncloud.ui');

/**
 * @class Zarafa.plugins.owncloud.ui.AttachFromOwncloudContentPanel
 * @extends Zarafa.core.ui.ContentPanel
 *
 * The content panel which shows the hierarchy tree of Owncloud account files.
 * @xtype Zarafa.plugins.owncloud.attachfromowncloudcontentpanel
 */
Zarafa.plugins.owncloud.ui.AttachFromOwncloudContentPanel = Ext.extend(Zarafa.core.ui.ContentPanel, {

	/**
	 * @cfg {Zarafa.core.data.IPMRecord} record The record to which
	 * we need to add attachment.
	 */
	record : null,

	/**
	 * @constructor
	 * @param config Configuration structure
	 */
	constructor : function(config)
	{
		config = config || {};
		Ext.applyIf(config, {
			layout					: 'fit',
			title					: container.getSettingsModel().get('zarafa/v1/plugins/owncloud/button_name') + _(' Attachment'),
			closeOnSave				: true,
			width					: 400,
			height					: 300,
			//Add panel
			items					: [
				{
					xtype			: 'Zarafa.plugins.owncloud.attachfromowncloudtreepanel',
					ref				: 'treePanel'
				}
			]
		});

		Zarafa.plugins.owncloud.ui.AttachFromOwncloudContentPanel.superclass.constructor.call(this, config);
	}
});

Ext.reg('Zarafa.plugins.owncloud.attachfromowncloudcontentpanel', Zarafa.plugins.owncloud.ui.AttachFromOwncloudContentPanel);Ext.namespace('Zarafa.plugins.owncloud.ui');

/**
 * @class Zarafa.plugins.owncloud.ui.SaveToOwncloudTreePanel
 * @extends Ext.tree.TreePanel
 * Shows tree of all user files from Owncloud
 * @xtype Zarafa.plugins.owncloud.savetoowncloudtreepanel
 */
Zarafa.plugins.owncloud.ui.SaveToOwncloudTreePanel = Ext.extend(Ext.tree.TreePanel, {

	selectedFolder : null,
	
	response : null,

	/**
	 * @constructor
	 * @param {Object} config
	 */
	constructor : function(config)
	{
		config = config || {};
		this.response = config.response;
		
		Ext.applyIf(config, {
			root: {
				nodeType: 'async',
				text: '/',
				id: '/',
				expanded : true
			},
			autoScroll: true,
			viewConfig: {
				style	:	{ overflow: 'auto', overflowX: 'hidden' }
			},
			maskDisabled: true,
			listeners: {
				click: function(n) {
					this.selectedFolder = n.attributes.id;
				}
			},
			loader : new Zarafa.plugins.owncloud.data.DirectoryLoader({loadfiles: false}),
			buttons : [
				this.createActionButtons()
			]
		});
		Zarafa.plugins.owncloud.ui.SaveToOwncloudTreePanel.superclass.constructor.call(this, config);
	},

	/**
	 * Creates action button for adding file
	 * from Owncloud to email as attachment
	 * @return {Array}
	 * @private
	 */
	createActionButtons : function()
	{
		return [
		{
			xtype   : 'button',
			text    : _('New Folder'),
			tooltip : {
				title   : _('New Folder'),
				text    : _('Create a new folder in owncloud')
			},
			width	: 100,
			handler : this.newFolder,
			scope   : this
		},
		{
			xtype   : 'button',
			text    : _('&nbsp;&nbsp;Save'),
			tooltip : {
				title   : _('Store Attachment'),
				text    : _('Store attachment to the selected owncloud folder.')
			},
			width	: 65,
			iconCls : 'icon_owncloud',
			handler : this.uploadFile,
			scope   : this
		}		
		];
	},

	/**
	 * Uploads the prepared attachment to Owncloud
	 * @private
	 */
	uploadFile : function()
	{
		if(this.selectedFolder == null) {
			Zarafa.common.dialogs.MessageBox.show({
				title   : _('Warning'),
				msg     : _('You have to choose a folder!'),
				icon    : Zarafa.common.dialogs.MessageBox.WARNING,
				buttons : Zarafa.common.dialogs.MessageBox.OK
			});
		} else {
			try {
				this.dialog.treePanel.disable();
				container.getRequest().singleRequest(
					'owncloudmodule',
					'uploadtooc',
					{
						tmpfile : this.response.tmpname,
						destdir : this.selectedFolder,
						destname : this.response.filename
					},
					new Zarafa.plugins.owncloud.data.ResponseHandler({
						successCallback : this.uploadDone.createDelegate(this)
					})
				);
			} catch (e) {
				Zarafa.common.dialogs.MessageBox.show({
					title   : _('Warning'),
					msg     : e.getMessage(),
					icon    : Zarafa.common.dialogs.MessageBox.WARNING,
					buttons : Zarafa.common.dialogs.MessageBox.OK
				});
			}
		}
	},
	
	/**
	 * Upload done =)
	 * @param {Object} response
	 * @private
	 */
	uploadDone : function(response)
	{
		if(response.status == 201) {
			container.getNotifier().notify('info.owncloud', 'Uploaded', 'Attachment successfully stored to owncloud!');
		} else {
			container.getNotifier().notify('error', 'Upload Failed', 'Attachment could not be stored in owncloud! Error: ' + response.status);
		}
		this.dialog.close();
	},	
	
	/**
	 * Creates a new owncloud folder
	 * @private
	 */
	newFolder : function()
	{
		if(this.selectedFolder == null) {
			Zarafa.common.dialogs.MessageBox.show({
				title   : _('Warning'),
				msg     : _('You have to choose a folder!'),
				icon    : Zarafa.common.dialogs.MessageBox.WARNING,
				buttons : Zarafa.common.dialogs.MessageBox.OK
			});
		} else {
			Zarafa.common.dialogs.MessageBox.prompt('Folder Name', 'Please enter a foldername', this.createRemoteFolder, this);
		}
	},
	
	createRemoteFolder : function(button, text)
	{
		if(button == "ok") {
			try {
				container.getRequest().singleRequest(
					'owncloudmodule',
					'createdir',
					{
						dirname : this.selectedFolder + text,
						basedir : this.selectedFolder
					},
					new Zarafa.plugins.owncloud.data.ResponseHandler({
						successCallback : this.createDirDone.createDelegate(this)
					})
				);
			} catch (e) {
				Zarafa.common.dialogs.MessageBox.show({
					title   : _('Warning'),
					msg     : e.getMessage(),
					icon    : Zarafa.common.dialogs.MessageBox.WARNING,
					buttons : Zarafa.common.dialogs.MessageBox.OK
				});
			}
		}
	},
	
	/**
	 * createDirDone =)
	 * @param {Object} response
	 * @private
	 */
	createDirDone : function(response)
	{
		if(response.status == 201) {
			container.getNotifier().notify('info.owncloud', 'Created', 'Directory created!');
			this.dialog.treePanel.getNodeById(response.basedir).reload();
		} else {
			container.getNotifier().notify('error', 'Creation Failed', 'Directory not created!');
		}
	}
});

Ext.reg('Zarafa.plugins.owncloud.savetoowncloudtreepanel', Zarafa.plugins.owncloud.ui.SaveToOwncloudTreePanel);
Ext.namespace('Zarafa.plugins.owncloud.ui');

/**
 * @class Zarafa.plugins.owncloud.ui.SaveToOwncloudContentPanel
 * @extends Zarafa.core.ui.ContentPanel
 *
 * The content panel which shows the hierarchy tree of Owncloud account files.
 * @xtype Zarafa.plugins.owncloud.savetoowncloudcontentpanel
 */
Zarafa.plugins.owncloud.ui.SaveToOwncloudContentPanel = Ext.extend(Zarafa.core.ui.ContentPanel, {

	/**
	 * @constructor
	 * @param config Configuration structure
	 */
	constructor : function(config)
	{
		config = config || {};

		Ext.applyIf(config, {
			layout					: 'fit',
			title					: container.getSettingsModel().get('zarafa/v1/plugins/owncloud/button_name') + _(' Attachment'),
			closeOnSave				: true,
			width					: 400,
			height					: 300,
			//Add panel
			items					: [
				{
					xtype			: 'Zarafa.plugins.owncloud.savetoowncloudtreepanel',
					ref				: 'treePanel',
					response		: config.record
				}
			]
		});

		Zarafa.plugins.owncloud.ui.SaveToOwncloudContentPanel.superclass.constructor.call(this, config);
	}

});

Ext.reg('Zarafa.plugins.owncloud.savetoowncloudcontentpanel' , Zarafa.plugins.owncloud.ui.SaveToOwncloudContentPanel);Ext.namespace('Zarafa.plugins.owncloud');

/**
 * @class Zarafa.plugins.owncloud.OwncloudAttachmentPlugin
 * @extends Zarafa.core.Plugin
 * This class is used for adding files from the users's Owncloud folder
 * to his eownclouds as attachments
 */
Zarafa.plugins.owncloud.AttachFromOwncloudPlugin = Ext.extend(Zarafa.core.Plugin, {

	/**
	 * @constructor
	 * @param {Object} config
	 */
	constructor : function(config)
	{
		config = config || {};
		Ext.applyIf(config, {
			name : 'attachfromowncloud'
		});
		Zarafa.plugins.owncloud.AttachFromOwncloudPlugin.superclass.constructor.call(this, config);
	},

	/**
	 * initialises insertion point for plugin
	 * @protected
	 */
	initPlugin : function()
	{
		Zarafa.plugins.owncloud.AttachFromOwncloudPlugin.superclass.initPlugin(this, arguments);
		
		/* load attachments from owncloud */
		this.registerInsertionPoint('main.attachment.method', this.onAttachmentInsertion)
		Zarafa.core.data.SharedComponentType.addProperty('common.dialog.attachments.owncloud');
	},

	/**
	 * Insert owncloud option in all attachment suggestions
	 */
	onAttachmentInsertion : function(include, btn) {
		return {
			text : container.getSettingsModel().get('zarafa/v1/plugins/owncloud/button_name'),
			handler : this.showOwncloudAttachmentDialog.createDelegate(this, [btn]),
			scope: this,
			iconCls: 'icon_owncloud'
		};
	},

	/**
	 * Initializes Dialog for adding attachment from Owncloud to eowncloud
	 * @param {Object} btn
	 * @private
	 */
	showOwncloudAttachmentDialog : function(btn)
	{
		Zarafa.core.data.UIFactory.openLayerComponent(Zarafa.core.data.SharedComponentType['common.dialog.attachments.owncloud'], btn.record, {
			title : _('Add attachment from ') + container.getSettingsModel().get('zarafa/v1/plugins/owncloud/button_name'),
			manager : Ext.WindowMgr
		});
	},
	
	/**
	 * Bid for the type of shared component
	 * and the given record.
	 * This will bid on calendar.dialogs.importevents
	 * @param {Zarafa.core.data.SharedComponentType} type Type of component a context can bid for.
	 * @param {Ext.data.Record} record Optionally passed record.
	 * @return {Number} The bid for the shared component
	 */
	bidSharedComponent : function(type, record)
	{
		var bid = -1;
		
		switch(type) {
			case Zarafa.core.data.SharedComponentType['common.dialog.attachments.owncloud']:
				if (record instanceof Zarafa.core.data.IPMRecord) {
					if (record.supportsAttachments()) {
						bid = 1;
					}
				}
				break;
		}
		return bid;
	},

	/**
	 * Will return the reference to the shared component.
	 * Based on the type of component requested a component is returned.
	 * @param {Zarafa.core.data.SharedComponentType} type Type of component a context can bid for.
	 * @param {Ext.data.Record} record Optionally passed record.
	 * @return {Ext.Component} Component
	 */
	getSharedComponent : function(type, record)
	{
		var component;

		switch(type) {
			case Zarafa.core.data.SharedComponentType['common.dialog.attachments.owncloud']:
				component = Zarafa.plugins.owncloud.ui.AttachFromOwncloudContentPanel;
				break;
		}

		return component;
	}
});Ext.namespace('Zarafa.plugins.owncloud');

/**
 * @class Zarafa.plugins.owncloud.OwncloudRcvAttachmentPlugin
 * @extends Zarafa.core.Plugin
 * This class is used for adding files from the users's Owncloud folder
 * to his eownclouds as attachments
 */
Zarafa.plugins.owncloud.SaveToOwncloudPlugin = Ext.extend(Zarafa.core.Plugin, {

	/**
	 * @constructor
	 * @param {Object} config
	 */
	constructor : function(config)
	{
		config = config || {};
		Ext.applyIf(config, {
			name : 'savetoowncloud'
		});
		Zarafa.plugins.owncloud.SaveToOwncloudPlugin.superclass.constructor.call(this, config);
	},

	/**
	 * initialises insertion point for plugin
	 * @protected
	 */
	initPlugin : function()
	{
		Zarafa.plugins.owncloud.SaveToOwncloudPlugin.superclass.initPlugin(this, arguments);
		
		/* store attachments in owncloud */
		this.registerInsertionPoint('common.contextmenu.attachment.actions', this.onAttachmentInsertion);
		Zarafa.core.data.SharedComponentType.addProperty('common.dialog.attachments.savetoowncloud');
	},
	
	/**
	 * called on successfull preload of attachment
	 */
	onSuccessPrepare : function(response)
	{
		Zarafa.common.dialogs.MessageBox.hide();
		Zarafa.core.data.UIFactory.openLayerComponent(Zarafa.core.data.SharedComponentType['common.dialog.attachments.savetoowncloud'], response, {
			title : _('Add attachment to ') + container.getSettingsModel().get('zarafa/v1/plugins/owncloud/button_name'),
			manager : Ext.WindowMgr
		});
	},

	/**
	 * Insert owncloud option in all attachment suggestions
	 */
	onAttachmentInsertion : function(include, btn) {
		return {
			text : container.getSettingsModel().get('zarafa/v1/plugins/owncloud/button_name'),
			handler : this.prepareAttachment.createDelegate(this, [btn,this.onSuccessPrepare]),
			scope: this,
			iconCls: 'icon_owncloud'
		};
	},

	/**
	 * prepareAttachment
	 * @param {Object} btn
	 * @private
	 */
	prepareAttachment : function(btn,callback)
	{
	
		Zarafa.common.dialogs.MessageBox.show({
			title: 'Please wait',
			msg: 'Loading attachment...',
			progressText: 'Initializing...',
			width:300,
			progress:true,
			closable:false
		});

		// progress bar... ;)
		var f = function(v){
			return function(){
				if(v == 100){
					Zarafa.common.dialogs.MessageBox.hide();
				}else{
					Zarafa.common.dialogs.MessageBox.updateProgress(v/100, Math.round(v)+'% loaded');
				}
		   };
		};
		
		for(var i = 1; i < 101; i++){
			setTimeout(f(i), 20*i);
		}
		
		/* store the attachment to a temporary folder and prepare it for uploading */
		var attachmentRecord = btn.records;
		var attachmentStore = attachmentRecord.store;
		
		var store = attachmentStore.getParentRecord().get('store_entryid');
		var entryid = attachmentStore.getAttachmentParentRecordEntryId();
		var attachNum = new Array(1);		
		if (attachmentRecord.get('attach_num') != -1)
			attachNum[0] = attachmentRecord.get('attach_num');
		else
			attachNum[0] = attachmentRecord.get('tmpname');
		var dialog_attachments = attachmentStore.getId();
		var filename = attachmentRecord.data.name;
		
		var responseHandler = new Zarafa.plugins.owncloud.data.ResponseHandler({
			successCallback: callback
		});
		
		// request attachment preperation
		container.getRequest().singleRequest(
			'owncloudmodule',
			'prepare-attachment',
			{
				entryid : entryid,
				store: store,
				attachNum: attachNum,
				dialog_attachments: dialog_attachments,
				filename: filename
			},
			responseHandler
		);
	},
	
	/**
	 * Bid for the type of shared component
	 * and the given record.
	 * This will bid on common.dialog.attachments.savetoowncloud
	 * @param {Zarafa.core.data.SharedComponentType} type Type of component a context can bid for.
	 * @param {Object} response Optionally passed response.
	 * @return {Number} The bid for the shared component
	 */
	bidSharedComponent : function(type, response)
	{
		var bid = -1;
		
		switch(type)
		{
			case Zarafa.core.data.SharedComponentType['common.dialog.attachments.savetoowncloud']:
				if(response.status == true) {
					bid = 1;
				}
				break;
		}
		return bid;
	},

	/**
	 * Will return the reference to the shared component.
	 * Based on the type of component requested a component is returned.
	 * @param {Zarafa.core.data.SharedComponentType} type Type of component a context can bid for.
	 * @param {Object} response Optionally passed response.
	 * @return {Ext.Component} Component
	 */
	getSharedComponent : function(type, response)
	{
		var component;

		switch(type)
		{
			case Zarafa.core.data.SharedComponentType['common.dialog.attachments.savetoowncloud']:
				component = Zarafa.plugins.owncloud.ui.SaveToOwncloudContentPanel;
				break;
		}

		return component;
	}
});Ext.namespace('Zarafa.plugins.owncloud.data');

/**
 * @class Zarafa.plugins.owncloud.data.ActionTypes
 * 
 * An enum that contains all possible Action types which
 * can be used on a {@link Zarafa.plugins.owncloud.OwncloudRecord OwncloudRecord} which
 * is being send to the server.
 * 
 * @singleton
 */
Zarafa.plugins.owncloud.data.ActionTypes =
{
	/**
	 * The file is attached to an email
	 * @property
	 * @type String
	 */
	ATTACHTOMAIL		: 'attachtomail',

	/**
	 * The file is deleted
	 * @property
	 * @type String
	 */
	DELETE	: 'delete',

	/**
	 * File information
	 * @property
	 * @type String
	 */
	INFO		: 'info'
};
Ext.namespace('Zarafa.plugins.owncloud.data');

/**
 * @class Zarafa.plugins.owncloud.data.DataModes
 * @extends Zarafa.core.Enum
 *
 * Enum containing the different data modes of the owncloud context. 
 * 
 * @singleton
 */
Zarafa.plugins.owncloud.data.DataModes = Zarafa.core.Enum.create({
	/**
	 * View all owncloud items from the selected folder(s).
	 *
	 * @property
	 * @type Number
	 */
	ALL : 0,
	/**
	 * View all found owncloud items in the selected folder(s).
	 *
	 * @property
	 * @type Number
	 */
	SEARCH : 1
});
Ext.namespace('Zarafa.plugins.owncloud.data');

/**
 * @class Zarafa.plugins.owncloud.data.FileTypes
 * @extends Zarafa.core.Enum
 *
 * Enum containing the different file types of the owncloud context. 
 * 
 * @singleton
 */
Zarafa.plugins.owncloud.data.FileTypes = Zarafa.core.Enum.create({
	/**
	 * Filetype: folder
	 *
	 * @property
	 * @type Number
	 */
	FOLDER : 0,
	/**
	 * Filetype: file
	 *
	 * @property
	 * @type Number
	 */
	FILE : 1
});
Ext.namespace('Zarafa.plugins.owncloud.data');

/**
 * @class Zarafa.plugins.owncloud.data.ViewModes
 * @extends Zarafa.core.Enum
 *
 * Enum containing the different viewing modes of the owncloud context.
 * 
 * @singleton
 */
Zarafa.plugins.owncloud.data.ViewModes = Zarafa.core.Enum.create({
	/**
	 * Don't show the preview panel
	 * @property
	 * @type Number
	 */
	NO_PREVIEW : 0,

	/**
	 * Show the preview panel to the right
	 * @property
	 * @type Number
	 */
	RIGHT_PREVIEW : 1,

	/**
	 * Show the preview panel in the bottom
	 * @property
	 * @type Number
	 */
	BOTTOM_PREVIEW : 2,

	/**
	 * Show the search results
	 * @property
	 * @type Number
	 */
	SEARCH : 3
});
Ext.namespace('Zarafa.plugins.owncloud.data');

/**
 * @class Zarafa.plugins.owncloud.data.Views
 * @extends Zarafa.core.Enum
 *
 * Enum containing the different views of the owncloud context. 
 * 
 * @singleton
 */
Zarafa.plugins.owncloud.data.Views = Zarafa.core.Enum.create({
	/**
	 * View all owncloud items from the selected folder(s) in the 'list' view.
	 *
	 * @property
	 * @type Number
	 */
	LIST : 0,

	/**
	 * View all found owncloud items from the selected folder(s) in the 'list' view.
	 *
	 * @property
	 * @type Number
	 */
	SEARCH : 1
});
Ext.namespace('Zarafa.plugins.owncloud.data');

/**
 * @class Zarafa.plugins.owncloud.data.SearchFields
 * @singleton
 *
 * Search fields for search bar
 */
Zarafa.plugins.owncloud.data.SearchFields = [{
	name : _('All text fields'),
	value : ''
}, {
	name : _('Modification Date'),
	value : ''
}, {
	name : _('Filename'),
	value : ''
}, {
	name : _('Filetype'),
	value : ''
}];
Ext.namespace('Zarafa.plugins.owncloud.data');

/**
 * @class Zarafa.plugins.owncloud.Actions
 * Common actions which can be used within {@link Ext.Button buttons}
 * or other {@link Ext.Component components} with action handlers.
 * @singleton
 */
Zarafa.plugins.owncloud.data.Actions = {

	/**
	 * Temporary Node, used in the create directory method
	 *
	 */
	tmpNode : undefined,

	/**
	 * The internal 'iframe' which is hidden from the user, which is used for downloading
	 * attachments. See {@link #doOpen}.
	 * @property
	 * @type Ext.Element
	 */
	downloadFrame : undefined,	
	
	/**
	 * Open a Panel in which the {@link Zarafa.core.data.IPMRecord record}
	 * can be viewed, or further edited.
	 *
	 * @param {Zarafa.core.data.IPMRecord} records The records to open
	 * @param {Object} config (optional) Configuration object used to create
	 * the Content Panel.
	 */
	openOwncloudContent : function(records, config)
	{
		var navpanel = Ext.getCmp("zarafa.owncloudtreepanel");
		
		Ext.each(records, function(record) {
			if(record.data.type == Zarafa.plugins.owncloud.data.FileTypes.FOLDER) {
				var ocontext = container.getCurrentContext();
				ocontext.model.store.loadPath(record.data.id);
				navpanel.getNodeById(record.data.id).reload();
			}
		});
	},
	
	/**
	 * Refreshes the left navigator tree
	 *
	 */
	refreshNavigatorTree : function()
	{
		var navpanel = Ext.getCmp("zarafa.owncloudtreepanel");
		var ocontext = container.getCurrentContext();
		
		// reload the navigation tree
		navpanel.getNodeById(ocontext.model.store.rootID).reload();
	},
	
	/**
	 * Download the selected items from owncloud
	 *
	 * @param {Array} ids An array of ids
	 */
	downloadItem : function(ids)
	{
		container.getNotifier().notify('info.owncloud', 'Downloading', 'Download started... please wait!');
		this.downloadSelectedFilesFromOwncloudToTmp(ids, function (items) {
			var downloadFrame = Ext.getBody().createChild({
				tag: 'iframe',
				cls: 'x-hidden'
			});
			
			Ext.each(items, function(item) {
				var url = 'plugins/owncloud/php/download_file.php?tmpname='+item.tmpname+'&filename='+item.name+'&secid='+item.secid+'&basedir='+item.basedir;
				downloadFrame.dom.contentWindow.location = url;
			});
		});
	},
	
	/**
	 * Create the upload form
	 *
	 * @param {Object} config
	 */
	createUploadDialog : function(model, config, path)
	{
		if(path == undefined) {
			path = model.store.rootID;
		}
	
		config = Ext.applyIf(config || {}, {
			modal : true,
			parentID: path
		});
		
		var componentType = Zarafa.core.data.SharedComponentType['zarafa.plugins.owncloud.uploadpanel'];
		Zarafa.core.data.UIFactory.openLayerComponent(componentType, undefined, config);
	},
	
	/**
	 * Sends a query with ids of selected files from user's Owncloud folder
	 * @private
	 */
	downloadSelectedFilesFromOwncloudToTmp : function(idsList, callback)
	{
		if(idsList.length >= 1) {
			try {
				container.getRequest().singleRequest(
					'owncloudlistmodule',
					'tmpdownload',
					{
						ids : idsList
					},
					new Zarafa.plugins.owncloud.data.ResponseHandler({
						successCallback : callback
					})
				);
			} catch (e) {
				Zarafa.common.dialogs.MessageBox.show({
					title   : _('Warning'),
					msg     : e.getMessage(),
					icon    : Zarafa.common.dialogs.MessageBox.WARNING,
					buttons : Zarafa.common.dialogs.MessageBox.OK
				});
			}
		}
	},

	/**
	 * Create a new Folder in {@link Zarafa.core.data.IPMRecord node}
	 *
	 * @param {Zarafa.core.data.IPMRecord} node The parent folder node
	 */
	createFolder : function(model, config, path)
	{
		if(path == undefined) {
			path = model.store.rootID;
		}
		Zarafa.common.dialogs.MessageBox.prompt(_('Folder Name'), _('Please enter a foldername'), function(button, text) {
			if(button == "ok") {
				
				try {
					var d = new Date();
					var nowUTC = d.getTime() + d.getTimezoneOffset()*60*1000;
					var data = {
						"filename" : text,
						"path" : path,
						"id" : path + text + "/", 
						"message_size" : "0",
						"lastmodified" : nowUTC,
						"type" : Zarafa.plugins.owncloud.data.FileTypes.FOLDER
					};
					var rec = new model.store.recordType(data);
					
					model.store.add(rec);
					model.store.on("update", this.refreshNavigatorTree, this);
					model.store.commitChanges();					
				} catch (e) {
					Zarafa.common.dialogs.MessageBox.show({
						title   : _('Warning'),
						msg     : e.getMessage(),
						icon    : Zarafa.common.dialogs.MessageBox.WARNING,
						buttons : Zarafa.common.dialogs.MessageBox.OK
					});
				}
			}			
		}, this);
	}
};
Ext.namespace('Zarafa.plugins.owncloud.data');

/**
 * @class Zarafa.plugins.owncloud.OwncloudRecordFields
 * Array of {@link Ext.data.Field field} configurations for the
 * {@link Zarafa.core.data.IPMRecord IPMRecord} object.
 * These fields will be available in all 'IPM.Owncloud' type messages.
 */
Zarafa.plugins.owncloud.data.OwncloudRecordFields = [
	{name: 'id'},
	{name: 'path'},
	{name: 'type', type: 'int', defaultValue: Zarafa.plugins.owncloud.data.FileTypes.FOLDER},
	{name: 'filename'},	
	{name: 'lastmodified', type: 'int', defaultValue: null},
	{name: 'message_size', type: 'int',	defaultValue: 0}
];

Zarafa.core.data.RecordCustomObjectType.addProperty('ZARAFA_OWNCLOUD');
Zarafa.core.data.RecordFactory.addFieldToMessageClass('IPM.Owncloud', Zarafa.plugins.owncloud.data.OwncloudRecordFields);
Zarafa.core.data.RecordFactory.setBaseClassToMessageClass('IPM.Owncloud', Zarafa.core.data.IPMRecord);
Zarafa.core.data.RecordFactory.setSubStoreToCustomType(Zarafa.core.data.RecordCustomObjectType.ZARAFA_OWNCLOUD, 'attachments', Zarafa.plugins.owncloud.data.AttachmentStore);Ext.namespace('Zarafa.plugins.owncloud.data');

/**
 * @class Zarafa.plugins.owncloud.data.OwncloudStore
 * @extends Zarafa.core.data.ListModuleStore
 * @xtype zarafa.owncloudstore
 * 
 * The OwncloudStore class provides a way to connect the 'owncloudlistmodule' in the server back-end to an 
 * Ext.grid.GridPanel object. It provides a means to retrieve owncloud listings asynchronously.
 * The store has to be initialised with a store Id, which corresponds (somewhat confusingly) to
 * a MAPI store id. The OwncloudStore object, once instantiated, will be able to retrieve and list
 * ownclouds from a single specific store only.
 * 
 * @constructor
 * @param {String} storeId a MAPI id that corresponds with a MAPI store on the server.
 */
Zarafa.plugins.owncloud.data.OwncloudStore = Ext.extend(Zarafa.core.data.ListModuleStore, {

	rootID : undefined,
	
	/**
	 * @constructor
	 * @param {String} storeId a MAPI id that corresponds with a MAPI store on the server.
	 */
	constructor : function()
	{
		this.rootID = "/";
		
		Zarafa.plugins.owncloud.data.OwncloudStore.superclass.constructor.call(this, {
			preferredMessageClass : 'IPM.Owncloud',
			autoSave: true,
			actionType : Zarafa.core.Actions['list'],
			defaultSortInfo : {
				field		: 'filename',
				direction	: 'asc'
			},
			baseParams : {
				id : this.rootID
			}
		});
	},
	
	loadPath : function(path)
	{
		this.rootID = path;
		var params = {
			id : this.rootID
		};
		this.load({params : params});
	}
});

Ext.reg('zarafa.owncloudstore', Zarafa.plugins.owncloud.data.OwncloudStore);
Ext.namespace('Zarafa.plugins.owncloud.ui');

/**
 * @class Zarafa.plugins.owncloud.ui.UploadButton
 * @extends Ext.SplitButton
 * @xtype zarafa.ownclouduploadbutton
 *
 * Special button which can be used for attaching items to a {@link Zarafa.core.data.IPMRecord IPMRecord}.
 * This utilizes the {@link #main.attachment.method} insertion point to allow plugins to register
 * alternative methods for attaching items to the record. These options will be shown inside the dropdown
 * list, while the default button action will be opening the Browsers File Selection dialog
 *
 * If the {@link Zarafa.core.plugins.RecordComponentUpdaterPlugin} is installed
 * in the {@link #plugins} array of this component, this component will automatically
 * load the {@link Zarafa.core.data.MAPIRecord record} into the component.
 * Otherwise the user of this component needs to call {@link #bindRecord}.
 */
Zarafa.plugins.owncloud.ui.UploadButton = Ext.extend(Ext.SplitButton, {
	/**
	 * @insert main.attachment.method
	 * Provide a new method for attaching files to a {@link Zarafa.core.data.IPMRecord IPMRecord}.
	 * This can be used by 3rd party plugins to insert a new MenuItem into the dropdown
	 * box for the {@link Zarafa.common.attachment.ui.AttachmentButton AttachmentButton}.
	 * This insertion point should return a {@link Ext.menu.Item item} instance of configuration
	 * @param {Zarafa.plugins.owncloud.ui.UploadButton} button This button
	 */
	 
	/**
	 * The {@link Zarafa.plugins.owncloud.OwncloudContextModel} which is obtained from the {@link #context}.
	 * @property
	 * @type Zarafa.mail.MailContextModel
	 */
	model : undefined,

	/**
	 * @constructor
	 * @param {Object} config Configuration object
	 */
	constructor : function(config)
	{
		config = config || {};
		if (!Ext.isDefined(config.model) && Ext.isDefined(config.context)) {
			config.model = config.context.getModel();
		}

		Ext.applyIf(config, {
			menu : {
				items : [{
					text : _('File upload'),
					//disabled : true,
					handler : function() { 
						Zarafa.plugins.owncloud.data.Actions.createUploadDialog(this.model);
					},
					scope: this
				}, {
					text: 'Create Folder', 
					action:'folder', 
					handler: function() { 
						Zarafa.plugins.owncloud.data.Actions.createFolder(this.model); 
					}, 
					scope : this
				}]
			},
			handler : function() {
					this.showMenu();
			},
			scope : this
		});
		Zarafa.plugins.owncloud.ui.UploadButton.superclass.constructor.call(this, config);
	},

	/**
	 * Event handler for opening the Browser's file selection dialog.
	 * See {@link #onFileInputChange} for the handling of the selected files.
	 * @private
	 */
	onFileUpload : function(field, event)
	{
		/*if (Zarafa.supportsFilesAPI()) {
			var attachEl = this.getAttachmentEl();

			// Register the change event handler
			// so we detect when the user selects a file.
			attachEl.on('change', this.onFileInputChange, this);

			// Mimick clicking on the <input> field
			// to open the File Selection dialog.
			attachEl.dom.click();
		} else {*/
			// With legacy upload we need a valid form, and we cannot
			// mimick clicking on the <input> field. Hence a secondary
			// dialog which should handle that.
			Zarafa.plugins.owncloud.data.Actions.createUploadDialog({
				parentID : "myroot",
				scope : this
			});
		//}
	}	
});

Ext.reg('zarafa.ownclouduploadbutton', Zarafa.plugins.owncloud.ui.UploadButton);
Ext.namespace('Zarafa.plugins.owncloud.ui');

/**
 * @class Zarafa.plugins.owncloud.ui.UploadPanel
 * @extends Zarafa.core.ui.ContentPanel
 * @xtype zarafa.plugins.owncloud.uploadpanel
 */
Zarafa.plugins.owncloud.ui.UploadPanel = Ext.extend(Zarafa.core.ui.ContentPanel, {
	/**
	 * @cfg {Function} callback The callback function which must be called when the
	 * {@link Ext.ux.form.FileUploadField File Input field} has been changed. It will
	 * receive the {@link Ext.form.BasicForm BasicForm} in which the input field is located
	 * as argument.
	 */
	callback : Ext.emptyFn,

	/**
	 * @cfg {Object} scope The scope for the {@link #callback} function
	 */
	scope : undefined,
	
	/**
	 * @cfg {String} parentID
	 */
	parentID : "/",
	
	/**
	 * @constructor
	 * @param {Object} config Configuration object
	 */
	constructor : function(config)
	{
		config = config || {};

		Ext.applyIf(config, {
			title : _('Select file'),
			layout : 'fit',
			width : 500,
			height : 75,
			items : [{
				xtype : 'form',
				layout: 'fit',
				fileUpload : true,
				padding : 5,
				items : [{
					xtype : 'fileuploadfield',
					buttonText: _('Browse') + '...',
					name : 'attachments',
					listeners : {
						'fileselected' : this.onFileSelected,
						'scope' : this
					}
				}],
				buttons : [{
					text : _('Cancel'),
					handler : this.close,
					scope : this
				}]
			}]
		});

		Zarafa.plugins.owncloud.ui.UploadPanel.superclass.constructor.call(this, config);
	},

	/**
	 * Event which is fired when the {@link Ext.ux.form.FileUploadField#fileselected fileselected} was fired.
	 * This will invoke the {@link #callback} function.
	 */
	onFileSelected : function(field)
	{
		var form = field.ownerCt.getForm();

		// Immediately hide the dialog, we will close
		// it (destroying the HTML elements) once the
		// upload has completed.
		this.hide();
		
		if (form.isValid()) {
			form.submit({
				waitMsg: 'Uploading files...',
				url: 'plugins/owncloud/php/upload_file.php',
				params: {
					"parentID" : this.parentID,
					"ocuser" : container.getSettingsModel().get('zarafa/v1/contexts/owncloud/username'),
					"ocpass" : container.getSettingsModel().get('zarafa/v1/contexts/owncloud/password'),
					"ocserv" : container.getSettingsModel().get('zarafa/v1/contexts/owncloud/server'),
					"ocbase" : container.getSettingsModel().get('zarafa/v1/contexts/owncloud/owncloud_path'),
					"ocssl" : container.getSettingsModel().get('zarafa/v1/contexts/owncloud/use_ssl'),
					"port" : container.getSettingsModel().get('zarafa/v1/contexts/owncloud/port'),
					"portssl" : container.getSettingsModel().get('zarafa/v1/contexts/owncloud/port_ssl')
				},
				failure: function(file, action) {
					Zarafa.common.dialogs.MessageBox.show({
						title   : _('Error'),
						msg     : _(action.result.errors[action.result.errors.type]),
						icon    : Zarafa.common.dialogs.MessageBox.ERROR,
						buttons : Zarafa.common.dialogs.MessageBox.OK
					});
				},
				success: function(file, action){
					var ocontext = container.getCurrentContext();
					ocontext.model.store.loadPath(action.result.parent);
				},
				scope : this
			});
		}

		form.on('actioncomplete', this.close, this, { delay : 5 });
		form.on('actionfailed', this.close, this, { delay : 5 });
	}
});

Ext.reg('zarafa.plugins.owncloud.uploadpanel', Zarafa.plugins.owncloud.ui.UploadPanel);
Ext.namespace('Zarafa.plugins.owncloud.ui');

/**
 * @class Zarafa.plugins.owncloud.ui.NavigatorTreePanel
 * @extends Ext.tree.TreePanel
 * Shows tree of all user files from Owncloud
 */
Zarafa.plugins.owncloud.ui.NavigatorTreePanel = Ext.extend(Ext.tree.TreePanel, {

	/**
	 * @constructor
	 * @param {Object} config
	 */
	constructor : function(config) {
		config = config || {};
		Ext.applyIf(config, {
			xtype : 'zarafa.owncloudtreepanel',
			ref : 'zarafa.owncloudtreepanel',
			id : 'zarafa.owncloudtreepanel',
			root: {
				nodeType: 'async',
				text: '/',
				id: '/',
				expanded : true
			},
			autoScroll: true,
			listeners: {
				click: function(n) {
					var ocontext = container.getCurrentContext();
					ocontext.model.store.loadPath(n.attributes.id);
					this.getNodeById(n.attributes.id).reload();
				},
				load: function(){
					/*var rootNode = this.getRootNode();
					for(var i = 0; i < rootNode.childNodes.length; i++) {
						if(!rootNode.childNodes[i].childrenRendered) {
							console.log('Child ' + i + ' is not rendered.');
						}
					}*/
				},
				scope: this
			},
			viewConfig: {
				style	:	{ overflow: 'auto', overflowX: 'hidden' }
			},
			maskDisabled: true,
			loader : new Zarafa.plugins.owncloud.data.DirectoryLoader({loadfiles: false})
		});
		Zarafa.plugins.owncloud.ui.NavigatorTreePanel.superclass.constructor.call(this, config);
	}
});

Ext.reg('zarafa.owncloudtreepanel',Zarafa.plugins.owncloud.ui.NavigatorTreePanel); 
Ext.namespace('Zarafa.plugins.owncloud.ui');

/**
 * @class Zarafa.plugins.owncloud.ui.OwncloudPreviewPanelToolbarButtons
 * @extends Object
 *
 * Contains special toolbar buttons for the previewpanel of the owncloud context.
 */
Zarafa.plugins.owncloud.ui.OwncloudPreviewPanelToolbarButtons = Ext.extend(Object, {
	/**
	 * @cfg {@link Zarafa.plugins.owncloud.OwncloudContextModel}
	 */
	model : undefined,

	/**
	 * @constructor
	 * @param config Configuration structure
	 */
	constructor : function(config){
		Ext.apply(this, config);
	},

	/**
	 * Function called when insertion point previewpanel.toolbar.right is called,
	 * Function returns configuration object for Copy/Move, Delete and Print buttons
	 * which are added on the right side of previewpanels toolbar.
	 * 
	 * @return {Object} Configuration object containing buttons
	 * which are added in the {@link Ext.Toolbar Toolbar}.
	 */
	getToolbarButtons : function()
	{
		return [
			// TODO: add Toolbar buttons
		];
	}
});
Ext.namespace('Zarafa.plugins.owncloud.ui');

/**
 * @class Zarafa.plugins.owncloud.ui.OwncloudToolbar
 * @extends Ext.Toolbar
 * The toolbar shown in the owncloud context
 */
Zarafa.plugins.owncloud.ui.OwncloudToolbar = Ext.extend(Ext.Toolbar, {

	/**
	 * @constructor
	 * @param {Object} config
	 */
	constructor : function(config)
	{
		config = config || {};
		
		config.plugins = Ext.value(config.plugins, []);
		config.plugins.push('zarafa.recordcomponentupdaterplugin');

		Ext.applyIf(config, {
			xtype : 'zarafa.owncloudtoolbar',
			items: [
			{
				xtype: 'zarafa.ownclouduploadbutton',
				plugins : [ 'zarafa.recordcomponentupdaterplugin' ],
				model: config.context.getModel(),
				overflowText: _('Create something new...'),
				tooltip: {
					title: _('New'),
					text: _('Upload File / Create Folder')
				},
				iconCls : 'icon_attachment'
			},
			{
				xtype: 'tbfill'
			},
			container.populateInsertionPoint('context.owncloud.toolbar.item', this),
			{
				xtype: 'tbseparator'
			},
			{
				xtype: 'tbtext', 
				text: 'OC ' + Zarafa.plugins.owncloud.data.Version.getOwncloudVersion()
			}			
			]
		});
		Zarafa.plugins.owncloud.ui.OwncloudToolbar.superclass.constructor.call(this, config);
	}
});

Ext.reg('zarafa.owncloudtoolbar',Zarafa.plugins.owncloud.ui.OwncloudToolbar);Ext.namespace('Zarafa.plugins.owncloud.ui');

/**
 * @class Zarafa.plugins.owncloud.ui.OwncloudPreviewPanel
 * @extends Zarafa.core.ui.PreviewPanel
 * @xtype zarafa.owncloudpreviewpanel
 */
Zarafa.plugins.owncloud.ui.OwncloudPreviewPanel = Ext.extend(Zarafa.core.ui.PreviewPanel, {
	/**
	 * @constructor
	 * @param config Configuration structure
	 */
	constructor : function(config)
	{
		config = config || {};

		if (!Ext.isDefined(config.model) && Ext.isDefined(config.context)) {
			config.model = config.context.getModel();
		}

		Ext.applyIf(config, {
			xtype: 'zarafa.owncloudpreviewpanel',
			cls: 'zarafa-previewpanel zarafa-context-mainpanel',
			recordComponentPluginConfig : Ext.applyIf(config.recordComponentPluginConfig || {}, {
				// Defer loading owncloud immediately, because the user
				// might be quickly clicking through the list of ownclouds.
				enableOpenLoadTask : true,
				autoOpenLoadTaskDefer : 250
			}),
			width: 600,
			height: 400,
			tbar : {
				height:33,
				items : [
					// TODO: add Toolbar items...
				]
			}
		});

		Zarafa.plugins.owncloud.ui.OwncloudPreviewPanel.superclass.constructor.call(this, config);
	}
});

Ext.reg('zarafa.owncloudpreviewpanel', Zarafa.plugins.owncloud.ui.OwncloudPreviewPanel);

Ext.namespace('Zarafa.plugins.owncloud.ui');

/**
 * @class Zarafa.plugins.owncloud.ui.OwncloudFileGridColumnModel
 * @extends Zarafa.common.ui.grid.ColumnModel
 *
 * The {@link Zarafa.plugins.owncloud.ui.OwncloudFileGridColumnModel OwncloudFileGridColumnModel}
 * is the default {@link Ext.grid.ColumnModel ColumnModel} containing two
 * different sets of {@link Ext.grid.Column columns}. The first set contains
 * all {@link Ext.grid.Column columns} which should be available in the
 * {@link Zarafa.plugins.owncloud.ui.OwncloudFileGrid OwncloudFileGrid} (either hidden by default,
 * or directly visible). For a more compact view, a more compact set is
 * provided. Switching between the two sets can be done using
 * {@link Zarafa.plugins.owncloud.ui.OwncloudFileGridColumnModel.useCompactView useCompactView}
 * during configuration, or {@link Zarafa.plugins.owncloud.ui.OwncloudFileGridColumnModel.setCompactView setCompactView}
 * when the {@link Zarafa.plugins.owncloud.ui.OwncloudFileGridColumnModel OwncloudFileGridColumnModel} is already active.
 */
Zarafa.plugins.owncloud.ui.OwncloudFileGridColumnModel = Ext.extend(Zarafa.common.ui.grid.ColumnModel, {
	/**
	 * @cfg {Boolean} useCompactView If true the compact column model will be
	 * used by default. Otherwise the default column model will be used which
	 * contains all possible columns.
	 */
	useCompactView : false,
	/**
	 * @constructor
	 * @param config Configuration structure
	 */
	constructor : function(config)
	{
		config = config || {};

		this.defaultColumns = this.createDefaultColumns();
		this.compactColumns = this.createCompactColumns();

		Ext.applyIf(config, {
			columns: this.defaultColumns,
			defaults: {
				sortable: true
			}
		});

		// Switch to compact view if needed
		if (config.useCompactView === true)
			config.columns = this.compactColumns;

		Ext.apply(this, config);

		Zarafa.plugins.owncloud.ui.OwncloudFileGridColumnModel.superclass.constructor.call(this, config);
	},

	/**
	 * Create an array of {@link Ext.grid.Column columns} which must be visible within
	 * the default view of this {@link Ext.grid.ColumnModel ColumnModel}.
	 *
	 * @return {Ext.grid.Column|Array} The array of columns
	 * @private
	 */
	createDefaultColumns : function()
	{
		return [{
			id : 'type',
			dataIndex : 'type',
			header : '<p class="icon_index">&nbsp;</p>',
			renderer : this.typeRenderer,
			width: 24,
			fixed : true,
			tooltip : _('Sort by: Type')
		},{
			header : 'ID',
			dataIndex : 'id',
			width: 50,
			hidden: true,
			tooltip : _('Sort by: ID')
		},{
			header : 'Path',
			dataIndex : 'path',
			width: 100,
			hidden: true,
			tooltip : _('Sort by: Path')
		},{
			header : _('File Name'),
			dataIndex : 'filename',
			width : 160,
			tooltip : _('Sort by: File Name')
		},{
			header : _('Last Modified'),
			dataIndex : 'lastmodified',
			width : 160,
			renderer : this.datetimeRenderer,
			tooltip : _('Sort by: Last Modified')
		},{
			header : _('Size'),
			dataIndex : 'message_size',
			width : 80,
			renderer : Ext.util.Format.fileSize,
			tooltip : _('Sort by: Size')
		}];
	},
	
	typeRenderer : function(value, p, record)
	{
		switch (value) {
			case Zarafa.plugins.owncloud.data.FileTypes.FOLDER: p.css = "x-tree-node-icon icon_folder_note"; break;
			case Zarafa.plugins.owncloud.data.FileTypes.FILE: p.css = "x-tree-node-icon icon_folder_task"; break;
			default : break;
		}

		// add extra css class for empty cell
		p.css += ' zarafa-grid-empty-cell';

		return '';
	},
	
	datetimeRenderer : function(value, p, record)
	{
		p.css = 'mail_date';

		// # TRANSLATORS: See http://docs.sencha.com/ext-js/3-4/#!/api/Date for the meaning of these formatting instructions
		value = new Date(value);
		return Ext.isDate(value) ? value.format(_('l d/m/Y G:i')) : _('None');
	},

	/**
	 * Create an array of {@link Ext.grid.Column columns} which must be visible within
	 * the compact view of this {@link Ext.grid.ColumnModel ColumnModel}.
	 *
	 * @return {Ext.grid.Column|Array} The array of columns
	 * @private
	 */
	createCompactColumns : function()
	{
		return [{
			id : 'column_type',
			dataIndex : 'type',
			header : '<p class="icon_index">&nbsp;</p>',
			renderer : this.typeRenderer,
			width: 24,
			fixed : true,
			tooltip : _('Sort by: Type')
		},{
			header : _('File Name'),
			dataIndex : 'filename',
			width : 160,
			tooltip : _('Sort by: File Name')
		},{
			header : _('Last Modified'),
			dataIndex : 'lastmodified',
			width : 100,
			renderer : Zarafa.common.ui.grid.Renderers.datetime,
			tooltip : _('Sort by: Last Modified')
		}];
	},

	/**
	 * This will switch the {@link Zarafa.plugins.owncloud.ui.OwncloudFileGridColumnModel columnmodel}
	 * configuration to either the compact or extended configuration.
	 *
	 * @param {Boolean} compact True to enable the compact view
	 */
	setCompactView : function(compact)
	{
		if (this.useCompactView !== compact) {
			this.useCompactView = compact;

			if (compact) {
				this.name = 'compact';
				// Extjs will store the this.columns into this.config after it has constructed
				// all the columns. At that point this.columns consists of the configuration objects,
				// while this.columns consists of all the allocated columns.
				this.defaultColumns = this.config;
				this.columns = this.compactColumns;
			} else {
				this.name = 'default';
				this.compactColumns = this.config;
				this.columns = this.defaultColumns;
			}

			this.setConfig(this.columns, false);
		}
	}
});
Ext.namespace('Zarafa.plugins.owncloud.ui');

/**
 * @class Zarafa.plugins.owncloud.ui.OwncloudFileGridContextMenu
 * @extends Zarafa.core.ui.menu.ConditionalMenu
 * @xtype zarafa.owncloudfilegridcontextmenu
 *
 * Extend {@link Zarafa.core.ui.menu.ConditionalMenu ConditionalMenu} to add the
 * {@link Zarafa.core.ui.menu.ConditionalItems ConditionalItems} for the
 * OwncloudContext.
 */
Zarafa.plugins.owncloud.ui.OwncloudFileGridContextMenu = Ext.extend(Zarafa.core.ui.menu.ConditionalMenu, {
	// Insertion points for this class
	/**
	 * @insert context.owncloud.contextmenu.actions
	 * Insertion point for adding actions menu items into the context menu
	 * @param {Zarafa.plugins.owncloud.ui.OwncloudGridContextMenu} contextmenu This contextmenu
	 */
	/**
	 * @insert context.owncloud.contextmenu.options
	 * Insertion point for adding options menu items into the context menu
	 * @param {Zarafa.plugins.owncloud.ui.OwncloudGridContextMenu} contextmenu This contextmenu
	 */

	/**
	 * @cfg {Zarafa.plugins.owncloud.OwncloudContext} context The context to which this panel belongs
	 */
	context : undefined,

	/**
	 * The {@link Zarafa.plugins.owncloud.OwncloudContextModel} which is obtained from the {@link #context}.
	 * @property
	 * @type Zarafa.plugins.owncloud.OwncloudContextModel
	 */
	model : undefined,

	/**
	 * @constructor
	 * @param {Object} config Configuration object
	 */
	constructor : function(config)
	{
		config = config || {};

		if (!Ext.isDefined(config.model) && Ext.isDefined(config.context)) {
			config.model = config.context.getModel();
		}

		Ext.applyIf(config, {
			items: [
				this.createContextActionItems(),
				{ xtype: 'menuseparator' },
				container.populateInsertionPoint('context.owncloud.contextmenu.actions', this),
				{ xtype: 'menuseparator' },
				container.populateInsertionPoint('context.owncloud.contextmenu.options', this)
			]
		});

		Zarafa.plugins.owncloud.ui.OwncloudFileGridContextMenu.superclass.constructor.call(this, config);
	},

	/**
	 * Create the Action context menu items
	 * @return {Zarafa.core.ui.menu.ConditionalItem|Array} The list of Action context menu items
	 * @private
	 */
	createContextActionItems : function()
	{
		return [{
			xtype: 'zarafa.conditionalitem',
			text : _('Open'),
			iconCls : 'icon_open',
			singleSelectOnly: true,
			handler: this.onContextItemOpen,
			scope: this
		},{
			xtype: 'zarafa.conditionalitem',
			text : _('Attach to mail'),
			//iconCls : 'icon_replyEowncloud',
			singleSelectOnly: true,
			responseMode : Zarafa.plugins.owncloud.data.ActionTypes.ATTACHTOMAIL,
			scope: this
		},{
			xtype: 'zarafa.conditionalitem',
			text : _('Delete'),
			iconCls : 'icon_delete',
			handler: this.onContextItemDelete,
			scope: this
		},{
			xtype: 'zarafa.conditionalitem',
			text : _('Info'),
			//iconCls : 'icon_forwardEowncloud',
			singleSelectOnly: true,
			responseMode : Zarafa.plugins.owncloud.data.ActionTypes.INFO,
			scope: this
		}];
	},

	/**
	 * Event handler which is called when the user selects the 'Open'
	 * item in the context menu. This will open the item in a new panel.
	 * @private
	 */
	onContextItemOpen : function()
	{
		Zarafa.plugins.owncloud.data.Actions.openOwncloudContent(this.records);
	},
	
	/**
	 * Event handler which is called when the user selects the 'Delete'
	 * item in the context menu. This will delete all selected records.
	 * @private
	 */
	onContextItemDelete : function()
	{
		var allowDelete = true;
		
		Ext.each(this.records, function(record) {
			if(record.data.id == (container.getSettingsModel().get('zarafa/v1/contexts/owncloud/owncloud_path') + "/") || record.data.filename == "..") {
				allowDelete = false;
			}
		}, this);
		
		if(allowDelete) {
			Zarafa.common.Actions.deleteRecords(this.records);
			Zarafa.plugins.owncloud.data.Actions.refreshNavigatorTree();
		}
		
	}
});

Ext.reg('zarafa.owncloudfilegridcontextmenu', Zarafa.plugins.owncloud.ui.OwncloudFileGridContextMenu);
Ext.namespace('Zarafa.plugins.owncloud.ui');

/**
 * @class Zarafa.plugins.owncloud.ui.OwncloudFileTable
 * @extends Ext.grid.GridPanel
 * The main gridpanel for our data
 */
Zarafa.plugins.owncloud.ui.OwncloudFileGrid = Ext.extend(Zarafa.common.ui.grid.GridPanel, {
	/**
	 * @cfg {Zarafa.mail.MailContext} context The context to which this panel belongs
	 */
	context : undefined,
	
	/**
	 * The {@link Zarafa.mail.MailContextModel} which is obtained from the {@link #context}.
	 * @property
	 * @type Zarafa.mail.MailContextModel
	 */
	model : undefined,
	
	/**
	 * @constructor
	 * @param {Object} config
	 */
	constructor : function(config)
	{
		config = config || {};
		
		if (!Ext.isDefined(config.model) && Ext.isDefined(config.context)) {
			config.model = config.context.getModel();
		}

		if (!Ext.isDefined(config.store) && Ext.isDefined(config.model)) {
			config.store = config.model.getStore();
		}
		config.store = Ext.StoreMgr.lookup(config.store);
		
		Ext.applyIf(config, {
			xtype : 'zarafa.owncloudfilegrid',
			ref : 'zarafa.owncloudfilegrid',
			id : 'zarafa.owncloudfilegrid',
			border : false,
			stateful : true,
			statefulRelativeDimensions : false,
			loadMask : this.initLoadMask(),
			viewConfig : this.initViewConfig(),
			sm : this.initSelectionModel(),
			cm : this.initColumnModel(),
			keys : {
				key : Ext.EventObject.DELETE,
				handler : this.onKeyDelete,
				scope : this
			}
		});
		Zarafa.plugins.owncloud.ui.OwncloudFileGrid.superclass.constructor.call(this, config);
		
	},
	
	/**
	 * Initialize event handlers
	 * @private
	 */
	initEvents : function()
	{
		Zarafa.mail.ui.MailGrid.superclass.initEvents.call(this);

		this.mon(this, 'cellcontextmenu', this.onCellContextMenu, this);
		this.mon(this, 'rowbodycontextmenu', this.onRowBodyContextMenu, this);
		//this.mon(this, 'cellclick', this.onCellClick, this);
		this.mon(this, 'rowdblclick', this.onRowDblClick, this);

		// Add a buffer to the following 2 event handlers. These are influenced by Extjs when a record
		// is removed from the store. However removing of records isn't performed in batches. This means
		// that wee need to offload the event handlers attached to removing of records in case that
		// a large batch of records is being removed.
		//this.mon(this.getSelectionModel(), 'rowselect', this.onRowSelect, this, { buffer : 1 });
		//this.mon(this.getSelectionModel(), 'selectionchange', this.onSelectionChange, this, { buffer : 1 });

		//this.mon(this.model, 'recordselectionchange', this.onRecordSelectionChange, this);

		this.mon(this.context, 'viewmodechange', this.onContextViewModeChange, this);
		this.onContextViewModeChange(this.context, this.context.getCurrentViewMode());
	},

	/**
	 * Initialize the {@link Ext.grid.GridPanel.loadMask} field
	 *
	 * @return {Ext.LoadMask} The configuration object for {@link Ext.LoadMask}
	 * @private
	 */
	initLoadMask : function()
	{
		return {
			msg : _('Loading files') + '...'
		};
	},

	/**
	 * Initialize the {@link Ext.grid.GridPanel#viewConfig} field
	 *
	 * @return {Ext.grid.GridView} The configuration object for {@link Ext.grid.GridView}
	 * @private
	 */
	initViewConfig : function()
	{
		return {
			// enableRowBody is used for enabling the rendering of
			// the second row in the compact view model. The actual
			// rendering is done in the function getRowClass.
			//
			// NOTE: Even though we default to the extended view,
			// enableRowBody must be enabled here. We disable it
			// later in onContextViewModeChange(). If we set false
			// here, and enable it later then the row body will never
			// be rendered. So disabling after initializing the data
			// with the rowBody works, but the opposite will not.
			enableRowBody : false,
			
			// We need a rowselector depth of 15 because of the nested
			// table in the rowBody.
			rowSelectorDepth : 15
		};
	},

	/**
	 * Initialize the {@link Ext.grid.GridPanel.sm SelectionModel} field
	 *
	 * @return {Ext.grid.RowSelectionModel} The subclass of {@link Ext.grid.AbstractSelectionModel}
	 * @private
	 */
	initSelectionModel : function()
	{
		return new Ext.grid.RowSelectionModel({
			singleSelect : false
		});
	},

	/**
	 * Initialize the {@link Ext.grid.GridPanel.cm ColumnModel} field.
	 *
	 * @return {Ext.grid.ColumnModel} The {@link Ext.grid.ColumnModel} for this grid
	 * @private
	 */
	initColumnModel : function()
	{
		return new Zarafa.plugins.owncloud.ui.OwncloudFileGridColumnModel();
	},
	
	/**
	 * Event handler which is fired when the {@link Zarafa.core.Context} fires the
	 * {@link Zarafa.core.Context#viewmodechange viewmodechange} event. This will check
	 * where the preview panel is located, and if needed change the
	 * {@link Ext.grid.Column columns} inside the {@link Ext.grid.ColumnModel ColumnModel}
	 * of the {@link Zarafa.plugins.owncloud.ui.OwncloudFileGrid OwncloudFileGrid}. Either use the extended (and more flexible)
	 * set or the more compact set.
	 *
	 * @param {Zarafa.core.Context} context The context which fired the event
	 * @param {Zarafa.plugins.owncloud.data.ViewModes} newViewMode The new active mode
	 * @param {Zarafa.plugins.owncloud.data.ViewModes} oldViewMode The previous mode
	 * @private
	 */
	onContextViewModeChange : function(context, newViewMode, oldViewMode)
	{
		if (newViewMode !== Zarafa.plugins.owncloud.data.ViewModes.SEARCH) {
			var compact = newViewMode === Zarafa.plugins.owncloud.data.ViewModes.RIGHT_PREVIEW;

			// The row body must only be enabled in compact view.
			this.getView().enableRowBody = compact;
			this.getColumnModel().setCompactView(compact);
		}
	},

	/**
	 * Event handler for the {@link #beforeconfigchange} event which is fired at the start of
	 * {@link Zarafa.common.ui.grid.ColumnModel#setConfig}. At this time we can
	 * check which columns should be activated based on the {@link #currentEntryId}.
	 *
	 * @param {Ext.gridColumnModel} columnModel The model which is being configured
	 * @param {Object} config The configuration object
	 * @private
	 */
	onBeforeConfigChange : function(columnModel, config)
	{
		if (!this.currentEntryId) {
			return;
		}
		
		// TODO: code...
		
		Zarafa.plugins.owncloud.ui.OwncloudFileGrid.superclass.onBeforeConfigChange.apply(this, arguments);
	},
	
	/**
	 * Event handler which is triggered when the user opems the context menu.
	 *
	 * There are some selection rules regarding the context menu. If no rows where
	 * selected, the row on which the context menu was requested will be marked
	 * as selected. If there have been rows selected, but the context menu was
	 * requested on a different row, then the old selection is lost, and the new
	 * row will be selected. If the row on which the context menu was selected is
	 * part of the previously selected rows, then the context menu will be applied
	 * to all selected rows.
	 *
	 * @param {Zarafa.plugins.owncloud.ui.OwncloudFile} grid The grid which was right clicked
	 * @param {Number} rowIndex The index number of the row which was right clicked
	 * @param {Number} cellIndex The index number of the column which was right clicked
	 * @param {Ext.EventObject} event The event structure
	 * @private
	 */
	onCellContextMenu : function(grid, rowIndex, cellIndex, event)
	{
		var sm = this.getSelectionModel();
		var cm = this.getColumnModel();

		if (sm.hasSelection()) {
			// Some records were selected...
			if (!sm.isSelected(rowIndex)) {
				// But none of them was the record on which the
				// context menu was invoked. Reset selection.
				sm.clearSelections();
				sm.selectRow(rowIndex);
			}
		} else {
			// No records were selected,
			// select row on which context menu was invoked
			sm.selectRow(rowIndex);
		}

		var column = {};
		// Take into account that the function onRowBodyContextMenu passes -1 as the column index.
		if(cellIndex >= 0){
			column = cm.getColumnById(cm.getColumnId(cellIndex));
		}
		var records = sm.getSelections();

		Zarafa.core.data.UIFactory.openDefaultContextMenu(records, { position : event.getXY(), context : this.context });
	},

	/**
	 * Event handler which is triggered when the user opems the context menu.
	 *
	 * This will call {@link onCellContextMenu} and pass -1 for the column index to prevent it 
	 * showing a special context menu if one would be set for specific columns.

	 *
	 * @param {Zarafa.plugins.owncloud.ui.OwncloudFile} grid The grid which was right clicked
	 * @param {Number} rowIndex The index number of the row which was right clicked
	 * @param {Ext.EventObject} event The event structure
	 * @private
	 */
	onRowBodyContextMenu : function(grid, rowIndex, event)
	{
		this.onCellContextMenu(grid, rowIndex, -1, event);
	},
	
	/**
	 * Event handler which is triggered when the user double-clicks on a particular item in the
	 * grid. This will update the store which 
	 * contains the selected item.
	 *
	 * @param {Grid} grid The Grid on which the user double-clicked
	 * @param {Number} rowIndex The Row number on which was double-clicked.
	 * @param {Ext.EventObject} e The event object
	 * @private
	 */
	onRowDblClick : function(grid, rowIndex, e)
	{
		var item = grid.store.getAt(rowIndex).data;
		var navpanel = Ext.getCmp("zarafa.owncloudtreepanel");
		
		if(item.type == Zarafa.plugins.owncloud.data.FileTypes.FOLDER) {
			this.store.loadPath(item.id);
			navpanel.getNodeById(item.id).reload();
		} else {
			Zarafa.plugins.owncloud.data.Actions.downloadItem(new Array(item.id));
		}
	},
	
	/**
	 * Event handler which is triggered when the user uses the delete button on a particular item in the
	 * grid. This will update the store which 
	 * contains the selected item.
	 *
	 * @param {Grid} grid The Grid on which the user double-clicked
	 * @param {Number} rowIndex The Row number on which was double-clicked.
	 * @param {Ext.EventObject} e The event object
	 * @private
	 */
	onKeyDelete : function(key, event)
	{
		var selections = this.getSelectionModel().getSelections();
		var allowDelete = true;
		
		if (!Ext.isEmpty(selections)) {
			Ext.each(selections, function(record) {
				if(record.data.id == (container.getSettingsModel().get('zarafa/v1/contexts/owncloud/owncloud_path') + "/") || record.data.filename == "..") {
					allowDelete = false;
				}
			}, this);
			
			if(allowDelete) {
				Zarafa.common.Actions.deleteRecords(selections);
				Zarafa.plugins.owncloud.data.Actions.refreshNavigatorTree();
			}
		}
	}
	
	// TODO: more eventhandlers...
});

Ext.reg('zarafa.owncloudfilegrid',Zarafa.plugins.owncloud.ui.OwncloudFileGrid);Ext.namespace('Zarafa.plugins.owncloud.ui');

/**
 * @class Zarafa.plugins.owncloud.ui.OwncloudMainPanel
 * @extends Zarafa.common.ui.ContextMainPanel
 * @xtype zarafa.owncloudmainpanel
 *
 * this class will be containing all the views that will be created for ownclouds folder
 * main panel for the owncloud content context
 */
Zarafa.plugins.owncloud.ui.OwncloudMainPanel = Ext.extend(Zarafa.common.ui.ContextMainPanel, {
	// Insertion points for this class
	/**
	 * @insert context.owncloud.toolbar.item
	 * Insertion point for populating Owncloud context's main toolbar.
	 * This item is only visible when this context is active.
	 * @param {Zarafa.plugins.owncloud.ui.OwncloudMainPanel} panel This panel
	 */
	/**
	 * @insert context.owncloud.toolbar.paging
	 *
	 * Insertion point for populating owncloud context's toolbar with extra
	 * pagination buttons. This can be used to replace the default {@link Ext.PagingToolbar}
	 * with an alternative. Owncloud that by default all paging toolbars will be visible, and
	 * hiding a particular toolbar is the responsibility of the new pagers.
	 * @param {Zarafa.plugins.owncloud.ui.OwncloudMainPanel} panel This panel
	 */
	/**
	 * @insert context.owncloud.views
	 * Insertion point for adding views within the main panel of owncloud context.
	 * This insertion point should be used in combination with 'main.maintoolbar.view.owncloud'
	 * insertion point, and also view should set its store in the config object, the reference of
	 * {@link Zarafa.plugins.owncloud.OwncloudContextModel OwncloudContextModel} is passed as parameter of this
	 * insertion point.
	 * @param {Zarafa.plugins.owncloud.ui.OwncloudMainPanel} mainpanel This mainpanel
	 * @param {Zarafa.plugins.owncloud.OwncloudContext} context The context for this panel
	 */
	 
	/**
	 * The main panel in which the various views are located.
	 * @property
	 * @type Zarafa.core.ui.SwitchViewContentContainer
	 */
	viewPanel : undefined,
	
	/**
	 * @constructor
	 * @param owncloudcontext
	 */
	constructor : function(config)
	{
		config = config || {};
		
		Ext.applyIf(config, {
			xtype : 'zarafa.owncloudmainpanel',
			layout : 'zarafa.switchborder',
			header : false,
			title : _('Files'),
			iconCls : 'icon_owncloud',
			items : [
				this.initFileGrid(config),
				this.initPreviewPanel(config.context)
			]
		});
		
		Zarafa.plugins.owncloud.ui.OwncloudMainPanel.superclass.constructor.call(this,config);
	},
	
	/**
	 * Initializes the {@link Zarafa.plugins.owncloud.ui.OwncloudFileGrid OwncloudFileGrid}
	 *
	 * @param {Object} config Configuration object
	 * @return {Zarafa.plugins.owncloud.ui.OwncloudFileGrid}
	 * @private
	 */
	initFileGrid : function(config)
	{
		return {
			xtype: 'panel',
			layout: 'zarafa.collapsible',
			cls : 'zarafa-context-mainpanel',
			minWidth : 200,
			minHeight: 200,
			region : 'center',
			collapsible : false,
			split : true,
			items: [{
				xtype : 'zarafa.searchbar',
				ref: '../searchBar',
				searchFields : Zarafa.plugins.owncloud.data.SearchFields,
				context : config.context
			},{
				xtype: 'zarafa.switchviewcontentcontainer',
				ref: '../viewPanel',
				layout : 'card',
				lazyItems : this.initViews(config.context)
			}],			
			tbar : {
				xtype: 'zarafa.owncloudtoolbar',
				title : _('Files'),
				iconCls : 'icon_owncloud',
				height:33,
				context : config.context
			}
		};
	},
	
	/**
	 * Function will initialize all views associated with owncloud context
	 * it will also get views added through 3rd party plugins and add it here
	 * @param {Zarafa.plugins.owncloud.OwncloudContextModel} model data part of owncloud context
	 * @return {Array} array of config objects of different views
	 * @private
	 */
	initViews : function(context)
	{
		// add the standard available views
		var allViews = [{
			xtype: 'zarafa.owncloudfilegrid',
			flex: 1,
			id    : 'owncloud-file-grid',
			anchor: '100%',
			context : context,
			ref: '../../owncloudGrid'
		}];

		var additionalViewItems = container.populateInsertionPoint('context.owncloud.views', this, context);
		allViews = allViews.concat(additionalViewItems);

		return allViews;
	},

	/**
	 * Initializes the {@link Zarafa.core.ui.PreviewPanel PreviewPanel}
	 *
	 * @param {Zarafa.mail.OwncloudContext} context The Owncloud Context
	 * @return {Zarafa.core.ui.PreviewPanel}
	 * @private
	 */
	initPreviewPanel : function(context)
	{
		return {
			xtype: 'zarafa.owncloudpreviewpanel',
			region : 'south',
			split : true,
			context: context
		};
	},

	/**
	 * Function called by Extjs when the panel has been {@link #render rendered}.
	 * At this time all events can be registered.
	 * @private
	 */
	initEvents : function()
	{
		if (Ext.isDefined(this.context)) {
			this.mon(this.context, 'viewchange', this.onViewChange, this);
			this.mon(this.context, 'viewmodechange', this.onViewModeChange, this);

			this.onViewChange(this.context, this.context.getCurrentView());
			this.onViewModeChange(this.context, this.context.getCurrentViewMode());
		}
	},

	/**
	 * Event handler which is fired when the currently active view inside the {@link #context}
	 * has been updated. This will update the call
	 * {@link #viewPanel}#{@link Zarafa.core.ui.SwitchViewContentContainer#switchView}
	 * to make the requested view active.
	 *
	 * @param {Zarafa.core.Context} context The context which fired the event.
	 * @param {Zarafa.plugins.owncloud.data.Views} newView The ID of the selected view.
	 * @param {Zarafa.plugins.owncloud.data.Views} oldView The ID of the previously selected view.
	 */
	onViewChange : function(context, newView, oldView)
	{
		switch (newView) {
			case Zarafa.plugins.owncloud.data.Views.LIST:
				this.viewPanel.switchView('owncloud-file-grid');
				break;
		}
	},

	/**
	 * Event handler which is fired when the {@link Zarafa.core.Context} fires the
	 * {@link Zarafa.core.Context#viewmodechange viewmodechange} event. This will
	 * convert the configured {@link Zarafa.plugins.owncloud.data.ViewModes mode} to a
	 * {@link Zarafa.common.ui.layout.SwitchBorderLayout.Orientation orientation}
	 * to be {@link Zarafa.common.ui.layout.SwitchBorderLayout.setOrientation applied}
	 * to the {@link #layout}.
	 * @param {Zarafa.core.Context} context The context which fired the event
	 * @param {Zarafa.plugins.owncloud.data.ViewModes} newViewMode The new active mode
	 * @param {Zarafa.plugins.owncloud.data.ViewModes} oldViewMode The previous mode
	 * @private
	 */
	onViewModeChange : function(context, newViewMode, oldViewMode)
	{
		var orientation;

		switch (newViewMode) {
			case Zarafa.plugins.owncloud.data.ViewModes.NO_PREVIEW:
				orientation = Zarafa.common.ui.layout.SwitchBorderLayout.Orientation.OFF;
				break;
			case Zarafa.plugins.owncloud.data.ViewModes.RIGHT_PREVIEW:
				orientation = Zarafa.common.ui.layout.SwitchBorderLayout.Orientation.HORIZONTAL;
				break;
			case Zarafa.plugins.owncloud.data.ViewModes.BOTTOM_PREVIEW:
				orientation = Zarafa.common.ui.layout.SwitchBorderLayout.Orientation.VERTICAL;
				break;
			case Zarafa.plugins.owncloud.data.ViewModes.SEARCH:
				return;
		}

		// This function could be called when the layout has not yet
		// been instantiated. In that case we update the layoutConfig
		// so it will be automatically picked up by the layout when
		// it needs it.
		var layout = this.getLayout();
		if (!Ext.isFunction(layout.setOrientation)) {
			if (Ext.isString(layout)) {
				this.layoutConfig = Ext.apply(this.layoutConfig || {}, { orientation : orientation });
			} else {
				this.layout.orientation = orientation;
			}
		} else {
			layout.setOrientation(orientation);
		}
	}
});
//register xtype of owncloud main panel
Ext.reg('zarafa.owncloudmainpanel',Zarafa.plugins.owncloud.ui.OwncloudMainPanel);
Ext.namespace('Zarafa.plugins.owncloud.context');

/**
 * @class Zarafa.plugins.owncloud.context.OwncloudContextModel
 * @extends Zarafa.core.ContextModel
 * class will instantiate {@link Zarafa.plugins.owncloud.OwncloudStore OwncloudStore} object
 */
Zarafa.plugins.owncloud.context.OwncloudContextModel = Ext.extend(Zarafa.core.ContextModel, {
	/**
	 * @constructor
	 * @param {Object} config Configuration object
	 */
	constructor : function(config)
	{
		config = config || {};
		
		if(!Ext.isDefined(config.store)) {
			config.store = new Zarafa.plugins.owncloud.data.OwncloudStore();
		}

		Zarafa.plugins.owncloud.context.OwncloudContextModel.superclass.constructor.call(this, config);
	},
	
	/**
	 * Create a new {@link Zarafa.core.data.IPMRecord IPMRecord}.
	 * @param {String} parentid id of the parent folder
	 * @return {Zarafa.core.data.IPMRecord} The new {@link Zarafa.core.data.IPMRecord IPMRecord}.
	 */
	createRecord : function(parentid)
	{
		parentid = parentid || "root";
		
		var record = Zarafa.core.data.RecordFactory.createRecordObjectByMessageClass('IPM.Owncloud', {
			store_entryid: "owncloud",
			parent_entryid: parentid
		});

		return record;
	},
	
	/**
	 * Event handler which is executed right before the {@link #datamodechange}
	 * event is fired. This allows subclasses to initialize the {@link #store}.
	 * This will apply a restriction to the {@link #store} if needed.
	 *
	 * @param {Zarafa.contact.ContactContextModel} model The model which fired the event.
	 * @param {Zarafa.contact.data.DataModes} newMode The new selected DataMode.
	 * @param {Zarafa.contact.data.DataModes} oldMode The previously selected DataMode.
	 * @private
	 */
	onDataModeChange : function(model, newMode, oldMode)
	{
		Zarafa.plugins.owncloud.context.OwncloudContextModel.superclass.onDataModeChange.call(this, model, newMode, oldMode);

		if (newMode !== oldMode && oldMode === Zarafa.plugins.owncloud.data.DataModes.SEARCH) {
			this.stopSearch();
		}

		switch (newMode) {
			case Zarafa.plugins.owncloud.data.DataModes.SEARCH:
				break;
			case Zarafa.plugins.owncloud.data.DataModes.ALL:
				this.load();
				break;
		}
	},

	/**
	 * Event handler for the {@link #searchstart searchstart} event.
	 * This will {@link #setDataMode change the datamode} to {@link Zarafa.plugins.owncloud.data.DataModes#SEARCH search mode}.
	 * The previously active {@link #getCurrentDataMode view} will be stored in the {@link #oldDataMode} and will
	 * be recovered when the {@link #onSearchStop search is stopped}.
	 * @param {Zarafa.core.ContextModel} model The model which fired the event
	 * @private
	 */
	onSearchStart : function(model)
	{
		if(this.getCurrentDataMode() != Zarafa.plugins.owncloud.data.DataModes.SEARCH){
			this.oldDataMode = this.getCurrentDataMode();
			this.setDataMode(Zarafa.plugins.owncloud.data.DataModes.SEARCH);
		}
	},
	
	/**
	 * Event handler for the {@link #searchstop searchstop} event.
	 * This will {@link #setDataMode change the datamode} to the {@link #oldDataMode previous datamode}.
	 * @param {Zarafa.core.ContextModel} model The model which fired the event
	 * @private
	 */
	onSearchStop : function(model)
	{
		if (this.getCurrentDataMode() === Zarafa.plugins.owncloud.data.DataModes.SEARCH) {
			this.setDataMode(this.oldDataMode);
		}
		delete this.oldDataMode;
	}
});
Ext.namespace('Zarafa.plugins.owncloud.context');

/**
 * @class Zarafa.plugins.owncloud.OwncloudContext
 * @extends Zarafa.core.Context
 *
 * This class will be used as a controller between {@link Zarafa.plugins.owncloud.OwncloudContextModel OwncloudContextModel}
 * and {@link Zarafa.plugins.owncloud.ui.OwncloudMainPanel OwncloudMainPanel}
 * Context that handles displaying owncloud type messages.
 */
Zarafa.plugins.owncloud.context.OwncloudContext = Ext.extend(Zarafa.core.Context, {
	// Insertion points for this class
	/**
	 * @insert main.maintoolbar.view.owncloud
	 * Insertion point for populating the main toolbar with a View button. This item is only visible
	 * when this context is active.
	 * @param {Zarafa.mail.OwncloudContext} context This context
	 */

	/**
	 * When searching, this property marks the {@link Zarafa.core.Context#getCurrentView view}
	 * which was used before {@link #onSearchStart searching started} the view was switched to
	 * {@link Zarafa.mail.data.Views#SEARCH}.
	 * @property
	 * @type Mixed
	 * @private
	 */
	oldView : undefined,

	/**
	 * When searching, this property marks the {@link Zarafa.core.Context#getCurrentViewMode viewmode}
	 * which was used before {@link #onSearchStart searching started} the viewmode was switched to
	 * {@link Zarafa.mail.data.ViewModes#SEARCH}.
	 * @property
	 * @type Mixed
	 * @private
	 */
	oldViewMode : undefined,
	 
	/**
	 * @constructor
	 * @param {Object} config configuration object
	 */
	constructor : function(config)
	{
		config = config || {};

		Ext.applyIf(config, {
			name : 'owncloudcontext',
			displayName : container.getSettingsModel().get('zarafa/v1/plugins/owncloud/button_name')
		});
				
		// register module names...
		this.registerModules();	

		// The tab in the top tabbar
		this.registerInsertionPoint('main.maintabbar.left', this.createMainTab, this);

		// The "New Owncloud" button which is available in all contexts
		this.registerInsertionPoint('main.maintoolbar.new.item', this.createNewOwncloudButton, this);

		Zarafa.plugins.owncloud.context.OwncloudContext.superclass.constructor.call(this, config);

		// add shared components
		Zarafa.core.data.SharedComponentType.addProperty('zarafa.plugins.owncloud.uploadpanel');
		
		// Add a tree control showing a list of owncloud folders to the navigation panel.
		// The control will be shown when the user selects the owncloud context from the button panel.
		this.registerInsertionPoint('navigation.center', this.createOwncloudNavigationPanel, this);
		
		// Register insertion point to insert toolbar buttons on the right side of menu.
		var toolbarButtons = new Zarafa.plugins.owncloud.ui.OwncloudPreviewPanelToolbarButtons({model: this.getModel()});
		this.registerInsertionPoint('previewpanel.toolbar.right', toolbarButtons.getToolbarButtons, toolbarButtons);
	},

	/**
	 * @return {Zarafa.plugins.owncloud.context.OwncloudContextModel} the owncloudt context model
	 */
	getModel : function()
	{
		if (!Ext.isDefined(this.model)) {
			this.model = new Zarafa.plugins.owncloud.context.OwncloudContextModel();
			this.model.on({
				'searchstart' : this.onModelSearchStart,
				'searchstop' : this.onModelSearchStop,
				scope : this
			});
		}
		return this.model;
	},
	
	/**
	 * Event handler for the {@link #model}#{@link Zarafa.core.ContextModel#searchstart searchstart} event.
	 * This will {@link #switchView switch the view} to {@link Zarafa.plugins.owncloud.data.Views#SEARCH search mode}.
	 * The previously active {@link #getCurrentView view} will be stored in the {@link #oldView} and will
	 * be recovered when the {@link #onModelSearchStop search is stopped}.
	 * @param {Zarafa.core.ContextModel} model The model which fired the event
	 * @private
	 */
	onModelSearchStart : function(model)
	{
		if(this.getCurrentView() != Zarafa.plugins.owncloud.data.Views.SEARCH && this.getCurrentViewMode() != Zarafa.plugins.owncloud.data.ViewModes.SEARCH){
			this.oldView = this.getCurrentView();
			this.oldViewMode = this.getCurrentViewMode();
			this.switchView(Zarafa.plugins.owncloud.data.Views.SEARCH, Zarafa.plugins.owncloud.data.ViewModes.SEARCH);
		}
	},

	/**
	 * Event handler for the {@link #model}#{@link Zarafa.core.ContextModel#searchstop searchstop} event.
	 * This will {@link #switchView switch the view} to the {@link #oldView previous view}.
	 * @param {Zarafa.core.ContextModel} model The model which fired the event
	 * @private
	 */
	onModelSearchStop : function(model)
	{
		this.switchView(this.oldView, this.oldViewMode);
		delete this.oldView;
		delete this.oldViewMode;
	},
	

	/**
	 * bid to get selected on Mapi folder selection
	 */
	bid : function(folder)
	{
		// Bid 1 when the folder is of the IPF.Owncloud type.
		if (folder.isContainerClass('IPF.Owncloud', true)) {
			return 1;
		}

		return -1;
	},

	/**
	 * Bid for the type of shared component and the given record.
	 * @param {Zarafa.core.data.SharedComponentType} type Type of component a context can bid for.
	 * @param {Ext.data.Record} record Optionally passed record.
	 * @return {Number} The bid for the shared component
	 */
	bidSharedComponent: function(type, record)
	{
		var bid = -1;

		if (Ext.isArray(record)) {
			record = record[0];
		}
		
		switch (type) {
			case Zarafa.core.data.SharedComponentType['zarafa.plugins.owncloud.uploadpanel']:
				bid = 1;
				break;
			case Zarafa.core.data.SharedComponentType['common.create']:
			case Zarafa.core.data.SharedComponentType['common.view']:
				if (record instanceof Zarafa.core.data.IPMRecord && record.isMessageClass('IPM.Owncloud', true)) {
					bid = 1;
				}
				break;
			case Zarafa.core.data.SharedComponentType['common.contextmenu']:				
				if (record instanceof Zarafa.core.data.IPMRecord && record.isMessageClass('IPM.Owncloud', true)) {
					bid = 1;
				}
				break;
			default : break;
		}
		return bid;
	},

	/**
	 * Will return the reference to the shared component.
	 * Based on the type of component requested a component is returned.
	 * @param {Zarafa.core.data.SharedComponentType} type Type of component a context can bid for.
	 * @param {Ext.data.Record} record Optionally passed record.
	 * @return {Ext.Component} Component
	 */
	getSharedComponent: function(type, record)
	{
		var component;
		switch (type) {
			case Zarafa.core.data.SharedComponentType['zarafa.plugins.owncloud.uploadpanel']:
				component = Zarafa.plugins.owncloud.ui.UploadPanel;
				break;
			case Zarafa.core.data.SharedComponentType['common.create']:
			case Zarafa.core.data.SharedComponentType['common.view']:
				component = Zarafa.plugins.owncloud.ui.OwncloudMainPanel;
				break;
			case Zarafa.core.data.SharedComponentType['common.contextmenu']:
				component = Zarafa.plugins.owncloud.ui.OwncloudFileGridContextMenu;
				break;
		}
		return component;
	},

	/**
	 * Creates the owncloud tree that is shown when the user selects the owncloud context from the
	 * button panel. It shows a tree of available owncloud folders that can be checked and unchecked.
	 * @private
	 */
	createOwncloudNavigationPanel : function()
	{
		return {
			xtype : 'zarafa.contextnavigation',
			context : this,
			items : [{
				xtype : 'panel',
				cls: 'zarafa-context-navigation-block',
				title : _('Owncloud'),
				items : [{ xtype: 'zarafa.owncloudtreepanel'}] // add the treeloader here
			}]
		};
	},

	/**
	 * creates a context panel
	 * @return configuration for owncloud context
	 * @private
	 */
	createContentPanel: function()
	{
		return {
			xtype : 'zarafa.owncloudmainpanel',
			context : this
		};
	},

	/**
	 * Create "New File" {@link Ext.menu.MenuItem item} for the "New item"
	 * {@link Ext.menu.Menu menu} in the {@link Zarafa.core.ui.MainToolbar toolbar}.
	 * This button should be shown in all {@link Zarafa.core.Context contexts} and
	 * is used to create a new Sticky Owncloud.
	 *
	 * @return {Object} The menu item for creating a new Sticky Owncloud item
	 */
	createNewOwncloudButton: function()
	{
		return {
			xtype: 'menuitem',
			text: _('Upload Owncloud File'),
			iconCls: 'icon_owncloud_category_large',
			newMenuIndex: 6,
			context: 'owncloud',
			//handler: function()
			//{
			//	Zarafa.plugins.owncloud.Actions.openCreateOwncloudContent(this.getModel());
			//},
			scope: this
		};
	},

	/**
	 * Returns the buttons for the dropdown list of the VIEW-button in the main toolbar. It will use the 
	 * main.maintoolbar.view.owncloud insertion point to allow other plugins to add their items at the end.
	 * 
	 * @return {Ext.Component|Array} an array of components
	 */
	getMainToolbarViewButtons : function()
	{
		var items = container.populateInsertionPoint('main.maintoolbar.view.owncloud', this) || [];
		
		var defaultItems = [{
			overflowText: _('No preview'),
			iconCls: 'icon_previewpanel_off',
			text: _('No preview'),
			valueView : Zarafa.plugins.owncloud.data.Views.LIST,
			valueViewMode : Zarafa.plugins.owncloud.data.ViewModes.NO_PREVIEW,
			valueDataMode : Zarafa.plugins.owncloud.data.DataModes.ALL,
			handler: this.onContextSelectView,
			scope: this
		},{
			overflowText: _('Right preview'),
			iconCls: 'icon_previewpanel_right',
			text: _('Right preview'),
			valueView : Zarafa.plugins.owncloud.data.Views.LIST,
			valueViewMode : Zarafa.plugins.owncloud.data.ViewModes.RIGHT_PREVIEW,
			valueDataMode : Zarafa.plugins.owncloud.data.DataModes.ALL,
			handler: this.onContextSelectView,
			scope: this
		},{
			overflowText: _('Bottom preview'),
			iconCls: 'icon_previewpanel_bottom',
			text: _('Bottom preview'),
			valueView : Zarafa.plugins.owncloud.data.Views.LIST,
			valueViewMode : Zarafa.plugins.owncloud.data.ViewModes.BOTTOM_PREVIEW,
			valueDataMode : Zarafa.plugins.owncloud.data.DataModes.ALL,
			handler: this.onContextSelectView,
			scope: this
		}];

		return defaultItems.concat(items);
	},

	/**
	 * Event handler which is fired when one of the View buttons
	 * has been pressed. This will call {@link #setView setView}
	 * to update the view.
	 * @param {Ext.Button} button The button which was pressed
	 * @private
	 */
	onContextSelectView : function(button)
	{
		this.getModel().setDataMode(button.valueDataMode);
		this.switchView(button.valueView, button.valueViewMode);
	},
	
	/**
	 * Adds a button to the top tab bar for this context.
	 * @return {Object} The button for the top tabbar 
	 * @private
	 */
	createMainTab: function()
	{
		return {
			text: this.getDisplayName(),
			tabOrderIndex: 7,
			context: this.getName()
		};
	},
	
	registerModules: function()
	{
		Zarafa.core.ModuleNames['IPM.OWNCLOUD'] = {
			list : 'owncloudlistmodule',
			item : 'owncloudlistmodule'
		}
	}
});
Ext.namespace('Zarafa.plugins.owncloud');

/**
 * @class Zarafa.plugins.owncloud.OwncloudPlugin
 * @extends Zarafa.core.Plugin
 * This cplugin is used to load the settings widget!
 */
Zarafa.plugins.owncloud.OwncloudPlugin = Ext.extend(Zarafa.core.Plugin, {
	/**
	 * @constructor
	 * @param {Object} config
	 */
	constructor : function(config)
	{
		config = config || {};
		
		Ext.applyIf(config, {
			name : 'owncloud',
			displayName : container.getSettingsModel().get('zarafa/v1/plugins/owncloud/button_name') + _(' Plugin'),
			about : Zarafa.plugins.owncloud.ABOUT
		});
		
		Zarafa.plugins.owncloud.OwncloudPlugin.superclass.constructor.call(this, config);
	},

	/**
	 * initialises insertion point for plugin
	 * @protected
	 */
	initPlugin : function()
	{
		Zarafa.plugins.owncloud.OwncloudPlugin.superclass.initPlugin.apply(this, arguments);
		
		/* Initialise Version Manager */
		Zarafa.plugins.owncloud.data.Version.init();
		
		/* settingswidget */
		/* Register the Mail category for the settings */
		this.registerInsertionPoint('context.settings.categories', this.createSettingCategories, this);
		/* use zarafa user as default user */
		if(container.getSettingsModel().get('zarafa/v1/contexts/owncloud/username') == "defaultusername") {
			container.getSettingsModel().set('zarafa/v1/contexts/owncloud/username', container.getUser().getUserName());
		}
	},
	
	/**
	 * Create the owncloud {@link Zarafa.settings.ui.SettingsCategory Settings Category}
	 * to the {@link Zarafa.settings.SettingsContext}. This will create new
	 * {@link Zarafa.settings.ui.SettingsCategoryTab tabs} for the
	 * {@link Zarafa.plugins.owncloud.settings.SettingsOwncloudCategory Owncloud} and the 
	 * in the {@link Zarafa.settings.ui.SettingsCategoryWidgetPanel Widget Panel}.
	 * @return {Array} configuration object for the categories to register
	 * @private
	 */
	createSettingCategories: function()
	{
		return [{
			xtype : 'Zarafa.plugins.owncloud.settingsowncloudcategory'
		}]
	}
});

/**
 * registers all plugins and the context
 */
Zarafa.onReady(function() {
	container.registerPlugin(new Zarafa.plugins.owncloud.OwncloudPlugin);
	
	if(container.getSettingsModel().get('zarafa/v1/plugins/owncloud/enable_attach_from_oc') === true)  {
		container.registerPlugin( new Zarafa.plugins.owncloud.AttachFromOwncloudPlugin);
	}
	if(container.getSettingsModel().get('zarafa/v1/plugins/owncloud/enable_store_to_oc') === true)  {
		container.registerPlugin( new Zarafa.plugins.owncloud.SaveToOwncloudPlugin);
	}
	if(container.getSettingsModel().get('zarafa/v1/plugins/owncloud/enable_browser') === true)  {
		container.registerContext(new Zarafa.plugins.owncloud.context.OwncloudContext());
	}
});
