<?php
/**
 * cachmanager.php, owncloud zarafa integration module
 *
 * Author: Christoph Haas <christoph.h@sprinternet.at>
 * Copyright (C) 2013 Christoph Haas
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
 
/** 
* This is a simple file caching class for webdav requests.
*/

include('../version.php');

class cache_manager {
	var $_debug = false;
	var $_cacheFile = "/tmp/wdquery.cache";
	var $_store = array();
	var $_enabled = false;
	
	/**
	 * Constructor
	 */
	function cache_manager($tmppath, $enabled) {
		$this->_cacheFile = $tmppath . session_id() . "wd.cache";
		$this->_enabled = $enabled;
		$this->_debug = $_ENV["OC_PLUGIN_DEBUG"];
		
		if(!is_writable($this->_cacheFile)) {
			$this->_enabled = false;
			$this->_error_log("Cachefile not writeable - disabling cache!");
		}
		
		$this->_error_log("cachefile: " . $this->_cacheFile);
		if(file_exists($this->_cacheFile)) {
			$this->_store = unserialize(file_get_contents($this->_cacheFile));
		}
	}
	
	/**
	 * Public method put
	 * 
	 * Puts a value for the identifier id to the cache store
	 * @param string id
	 * @param string value
	 * @access public
	 */
	function put($id, $value) {
		if($this->_enabled) {
			$hid = $this->hash($id);
			
			
			if(!$this->is_cached($id)) {
				$this->_store[$hid] = $value;
				$this->flush_store();
				$this->_error_log('put ' . $id . ' to cache with hid: ' . $hid);
			} else {
				$this->_error_log($id . ' already in cache');
			}
		}
	}
	
	/**
	 * Public method get
	 * 
	 * Returns the value for id or null if id is not cached
	 * @param string id
	 * @return array. NULL if not cached
	 * @access public
	 */
	function get($id) {
		$ret = NULL;
		
		if($this->_enabled) {
			$hid = $this->hash($id);
			
			if($this->is_cached($id)) {
				$ret = $this->_store[$hid];
			} else {
				$this->_error_log('requested id: ' . $id . ' is not in the cache');
			}
		}
		return $ret;
	}
	
	/**
	 * Public method remove
	 * 
	 * Removes a cached value from the store
	 * @param string id
	 * @access public
	 */
	function remove($id) {
		if($this->_enabled) {
			$hid = $this->hash($id);
			
			if($this->is_cached($id)) {
				unset($this->_store[$hid]);
				$this->flush_store();
			} else {
				$this->_error_log('requested id: ' . $id . ' is not in the cache - nothing to be removed');
			}
		}
	}
	
	/**
	 * Public method clear
	 * 
	 * Clears the whole cache
	 * @access public
	 */
	function clear() {
		if($this->_enabled) {
			unset($this->_store);
			$this->_store = array();
			$this->flush_store();
		}
	}
	
	/**
	 * Private method is_cached
	 * 
	 * Puts a value for the identifier id to the cache store
	 * @param string id
	 * @return boolean
	 * @access private
	 */
	function is_cached($id) {
		if($this->_enabled) {
			$hid = $this->hash($id);
			
			return array_key_exists($hid, $this->_store);
		} else {
			return false;
		}
	}
	
	/**
	 * Private method flush_store
	 * 
	 * Flushes changes to disk
	 * @access private
	 */
	function flush_store() {
		if(is_writable($this->_cacheFile)) {
			file_put_contents($this->_cacheFile, serialize($this->_store));
		} else {
			$this->_error_log("Cachefile not writeable!!");
		}
	}
	
	/**
	 * Private method hash
	 * 
	 * Hashes the given id to a unique string
	 * @param string id
	 * @return string
	 * @access private
	 */
	function hash($id) {
		return base64_encode($id);
	}
	
	/**
	 * Private method _error_log
	 * 
	 * a simple php error_log wrapper. 
	 * @param string err_string
	 * @access private
	 */
	function _error_log($err_string) {
		if ($this->_debug) {
			error_log($err_string);
		}
	}
}
 ?>