<?php
/**
 * class.helper.php, owncloud zarafa integration module
 *
 * Author: Christoph Haas <christoph.h@sprinternet.at>
 * Copyright (C) 2012-2013 Christoph Haas
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
 
 /**
  * Some core functionalities used in all other files
  */

  
class Helper {
	
	/**
	 * get_mime
	 * 
	 * Returns the mimetype for the specified file
	 * @param filename 
	 * @param mode 0 = full check, 1 = extension check only
	 * @return string mimetype
	 */
	static function get_mime($filename, $mode=0) {
		// mode 0 = full check
		// mode 1 = extension check only

		$mime_types = array(
			'txt' => 'text/plain',
			'htm' => 'text/html',
			'html' => 'text/html',
			'php' => 'text/html',
			'css' => 'text/css',
			'js' => 'application/javascript',
			'json' => 'application/json',
			'xml' => 'application/xml',
			'swf' => 'application/x-shockwave-flash',
			'flv' => 'video/x-flv',

			// images
			'png' => 'image/png',
			'jpe' => 'image/jpeg',
			'jpeg' => 'image/jpeg',
			'jpg' => 'image/jpeg',
			'gif' => 'image/gif',
			'bmp' => 'image/bmp',
			'ico' => 'image/vnd.microsoft.icon',
			'tiff' => 'image/tiff',
			'tif' => 'image/tiff',
			'svg' => 'image/svg+xml',
			'svgz' => 'image/svg+xml',

			// archives
			'zip' => 'application/zip',
			'rar' => 'application/x-rar-compressed',
			'exe' => 'application/x-msdownload',
			'msi' => 'application/x-msdownload',
			'cab' => 'application/vnd.ms-cab-compressed',

			// audio/video
			'mp3' => 'audio/mpeg',
			'qt' => 'video/quicktime',
			'mov' => 'video/quicktime',
			'mp4' => 'video/mp4',
			'webm' => 'video/webm',

			// adobe
			'pdf' => 'application/pdf',
			'psd' => 'image/vnd.adobe.photoshop',
			'ai' => 'application/postscript',
			'eps' => 'application/postscript',
			'ps' => 'application/postscript',

			// ms office
			'doc' => 'application/msword',
			'rtf' => 'application/rtf',
			'xls' => 'application/vnd.ms-excel',
			'ppt' => 'application/vnd.ms-powerpoint',
			'docx' => 'application/msword',
			'xlsx' => 'application/vnd.ms-excel',
			'pptx' => 'application/vnd.ms-powerpoint',

			// open office
			'odt' => 'application/vnd.oasis.opendocument.text',
			'ods' => 'application/vnd.oasis.opendocument.spreadsheet',
		);

		$ext = strtolower(array_pop(explode('.',$filename)));

		if(function_exists('mime_content_type')&&$mode==0){ 
			$mimetype = mime_content_type($filename); 
			return $mimetype;
		}elseif(function_exists('finfo_open')&&$mode==0){ 
			$finfo = finfo_open(FILEINFO_MIME); 
			$mimetype = finfo_file($finfo, $filename); 
			finfo_close($finfo); 
			return $mimetype; 
		}elseif(array_key_exists($ext, $mime_types)){ 
			return $mime_types[$ext]; 
		}else { 
			return 'application/octet-stream'; 
		} 
	}
	
	/**
	 * return size to human readable filesize
	 * @param {Int} $filesize
	 * @param {Int} $fractional digits
	 */
	static function human_filesize($bytes, $decimals = 2) {
		$sz = ' KMGTP';
		$factor = floor((strlen($bytes) - 1) / 3);
		return sprintf("%.{$decimals}f", $bytes / pow(1024, $factor)) . " " . @$sz[$factor] . "B";
	}
	
	/**
	 * @param {String} $path
	 * @return {String} filename
	 * @private
	 */
	static function getFilenameFromPath($path) {
		$pathParts = explode('/', $path);
		return end($pathParts);
	}
	
	/** 
	 * creates a random string
	 *
	 * @param {Integer} length The length of the random string
	 */
	static function randomstring($length = 6) {
		// $chars - all allowed charakters
		$chars = "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ1234567890";

		srand((double)microtime()*1000000);
		$i = 0;
		while ($i < $length) {
			$num = rand() % strlen($chars);
			$tmp = substr($chars, $num, 1);
			$pass = $pass . $tmp;
			$i++;
		}
		return $pass;
	}
	
	/** 
	 * creates a security file, which is checked before downloading a file
	 *
	 * @param {String} secid A random id
	 */
	static function createSecIDFile($basepath, $secid) {
		$lockFile = $basepath . DIRECTORY_SEPARATOR . "secid." . $secid;
		$fh = fopen($lockFile, 'w') or die("can't open secid file");
		$stringData = date(DATE_RFC822);
		fwrite($fh, $stringData);
		fclose($fh);
	}
	
	/**
	 * form the structure needed for frontend
	 * for the list of folders and files
	 * @param basedir
	 * @param dir
	 * @param data
	 * @param navtree parse for navtree or browser
	 * @return {Array} nodes for current path folder
	 * @private
	 */
	static function parseFolderContent($basedir, $dir, $data, $onlyfolders = false, $navtree = false) {
		$nodes = array();
		$oc_base = $GLOBALS["settings"]->get("zarafa/v1/contexts/owncloud/owncloud_path");
		
		$parentdir =  dirname($basedir);  // get parent dir
		if($parentdir != "/") {
			$parentdir = $parentdir . "/";
		}		
		
		foreach($dir as $node) {
			$type = 1;
			
			$id = substr(urldecode($node['href']),strlen($oc_base));
			
			if($_ENV["OC_PLUGIN_DEBUG"]) {
				error_log("parsing: " . $id . " in base: " . $basedir);
			}
			
			$filename =  basename($id);
			$basedir2 = $basedir."/";
			if (strcmp($id,$basedir) == 0 || strcmp($id,$basedir2) == 0 ) { // current dir...
				continue;
			}
			if (strcmp($node['resourcetype'],"collection") == 0) { // we have a folder
				$type = 0;
			}
			
			if($navtree) {
				if ($type == 0) { // we have a folder
					$nodes[$id] = array(
						'id'	=>	$id,
						'path'	=>	dirname($id),
						'text'	=>	$filename,
						'filename' => $filename,
						'leaf'	=>	false,
						'filename'=>$filename
					);
				} else {
					if(!$onlyfolders) {	// skip files if $onlyfolders == true
						$nodes[$id] = array(
							'id'	=>	$id,
							'path'	=>	dirname($id),
							'text'	=>	$filename . '(' . Helper::human_filesize(intval($node['getcontentlength'])) . ')',
							'filesize' => intval($node['getcontentlength']),
							'filename' => $filename,
							'leaf'	=>	true,
							'checked'=>	false
						);
					}
				}
			} else {			
				$nodes[$id] = array('props' =>
					array(
						'id'	=>	urldecode($id),
						'path'	=>	dirname(urldecode($id)),
						'filename'=>$filename,							
						'message_size'	=>	$node['getcontentlength'] == null ? 0 : intval($node['getcontentlength']),
						'lastmodified' => strtotime($node['lastmodified']) * 1000,
						'message_class' => "IPM.Owncloud",
						'type'=> $type
					),
					'entryid' => urldecode($id),
					'store_entryid' => 'owncloud',
					'parent_entryid' => dirname($id) . "/"
				);
			}
		}
		
		$sortkey = "filename";
		$sortdir = "ASC";
		
		if(isset($data['sort'])) {
			$sortkey = $data['sort'][0]['field'];
			$sortdir = $data['sort'][0]['direction'];
		}
		
		if($_ENV["OC_PLUGIN_DEBUG"]) {
			error_log("sorting by " . $sortkey . " in direction: " . $sortdir);
		}
		
		$nodes = Helper::sort_by_key($navtree, $nodes, $sortkey, $sortdir);
		
		if(!$navtree && $basedir != "/") {	// add a back button (browser only)
			$nodes = array($parentdir => 
				array('props' =>
					array(
						'id'	=>	urldecode($parentdir),
						'path'	=>	dirname(urldecode($parentdir)),
						'filename'=> "..",							
						'message_size'	=>	0,
						'lastmodified' => null,
						'message_class' => "IPM.Owncloud",
						'type'=> 0
					),
					'entryid' => urldecode($parentdir),
					'store_entryid' => 'owncloud',
					'parent_entryid' => dirname(urldecode($parentdir)) . "/"
				)
			) + $nodes;
		}
		
		return array_values($nodes);
	}
	
	/**
	 * Sort multidimensional array by any key
	 *
	 * @param navtree if we have a navtree, we have to sort different fields!
	 * @param array to sort
	 * @param key 
	 * @param dir ASC or DESC
	 */
	static function sort_by_key ($navtree, $arr, $key, $dir) { 
		global $key2sort; 
		$key2sort = $key; 
		
		$prefix = "";
		if($navtree){
			$prefix = "nav";
		}
		
		if($dir == "DESC")
			usort($arr, array('self', $prefix . 'invpropsort'));
		else
			usort($arr, array('self', $prefix . 'propsort'));
		return ($arr); 
	} 
	
	/**
	 * compare function for multidimensional array sorting
	 *
	 * @param a
	 * @param b 
	 */
	static function propsort ($a, $b) {
		global $key2sort;
		
		if($a['props']['type'] == $b['props']['type']) {
			return (strcasecmp ($a['props'][$key2sort],$b['props'][$key2sort]));
		}
		
		return ($a['props']['type'] - $b['props']['type']);
	}
	
	/**
	 * inverse compare function for multidimensional array sorting
	 *
	 * @param a
	 * @param b 
	 */
	static function invpropsort ($a, $b) {
		global $key2sort;
		
		
		
		if($a['props']['type'] == $b['props']['type']) {
			return (-1 * strcasecmp ($a['props'][$key2sort],$b['props'][$key2sort]));
		}
		
		return ($a['props']['type'] - $b['props']['type']);
	}
	
	/**
	 * compare function for multidimensional array sorting (navtree)
	 *
	 * @param a
	 * @param b 
	 */
	static function navpropsort ($a, $b) {
		global $key2sort;
		
		if($a['leaf'] == $b['leaf']) {
			return (strcasecmp ($a[$key2sort],$b[$key2sort]));
		}
		
		return ((int) $a['leaf'] - (int) $b['leaf']);
	}
	
	/**
	 * inverse compare function for multidimensional array sorting (navtree)
	 *
	 * @param a
	 * @param b 
	 */
	static function navinvpropsort ($a, $b) {
		global $key2sort;
		
		
		
		if($a['leaf'] == $b['leaf']) {
			return (-1 * strcasecmp ($a[$key2sort],$b[$key2sort]));
		}
		
		return ((int) $a['leaf'] - (int) $b['leaf']);
	}
}
?>