<?php
/**
 * classowncloudbrowsermodule.php, owncloud zarafa integration module
 *
 * Author: Christoph Haas <christoph.h@sprinternet.at>
 * Copyright (C) 2012-2013 Christoph Haas
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
 
include('version.php');
include('Owncloud/class.helper.php');
require_once('Owncloud/webdav.php');
require_once('Owncloud/authexception.php');
require_once('Owncloud/cachemanager.php');

/**
 * Owncloud Module
 */
class OwncloudBrowserModule extends ListModule {

	private $wdc;				// webdav client
	private $cache;				// cache manager
	private $initdone = false;

	/**
	 * Constructor
	 * @param int $id unique id.
	 * @param array $data list of all actions.
	 */
	function OwncloudBrowserModule($id, $data) {
		parent::ListModule($id, $data);

		$this->start = 0;
	}

	/**
	 * Creates the notifiers for this module,
	 * and register them to the Bus.
	 */
	/*function createNotifiers() {
		$entryid = $this->getEntryID();
		$GLOBALS["bus"]->registerNotifier('owncloudlistnotifier', $entryid);
	}*/

/**
	 * Initialize PHP API of Owncloud
	 * @param $data
	 */
	private function bootstrap($data) {
		$this->wdc = new webdav_client();
		$oc_base = $GLOBALS["settings"]->get("zarafa/v1/contexts/owncloud/owncloud_path");
		$oc_server = $GLOBALS["settings"]->get("zarafa/v1/contexts/owncloud/server");
		$oc_ssl = $GLOBALS["settings"]->get("zarafa/v1/contexts/owncloud/use_ssl");
		
		$this->wdc->set_server($oc_server);
		if($oc_ssl == true) {
			$this->wdc->set_port(intval($GLOBALS["settings"]->get("zarafa/v1/contexts/owncloud/port_ssl")));
		} else {
			$this->wdc->set_port(intval($GLOBALS["settings"]->get("zarafa/v1/contexts/owncloud/port")));
		}
		$this->wdc->set_ssl($oc_ssl);
		$this->wdc->set_pathprefix($oc_base);
		// use HTTP/1.1
		$this->wdc->set_protocol(1);
		// enable debugging
		$this->wdc->set_debug($_ENV["OC_PLUGIN_DEBUG"]);
		
		// initialize the cachemanager
		$this->cache = new cache_manager(PLUGIN_OWNCLOUDBROWSER_TMP, $GLOBALS["settings"]->get("zarafa/v1/contexts/owncloud/enable_caching"));
	}

	/**
	 * Executes all the actions in the $data variable.
	 * Exception part is used for authentication errors also
	 * @return boolean true on success or false on failure.
	 */
	public function execute() {
		$result = false;

		foreach($this->data as $actionType => $actionData)
		{			
			/* we have to fix this.... */
			//TODO: rewrite this dirty code...
			if(!$this->initdone && isset($actionData)) {
				$this->bootstrap($actionData);
				$this->initdone = true;
			}
			if(isset($actionType)) {
				try {
					if($_ENV["OC_PLUGIN_DEBUG"]) {
						error_log("exec: " . $actionType);
					}
					switch($actionType)
					{
						case "list":
							$result = $this->loadNode($actionType, $actionData);
							break;
						case "delete":
							$result = $this->delete($actionType, $actionData);
							break;
						case "open":
							$result = $this->open($actionType, $actionData);
							break;
						case "save":
							$result = $this->createDirectory($actionType, $actionData);
							break;
						default:
							$this->handleUnknownActionType($actionType);
					}

				} catch (MAPIException $e) {
					if($_ENV["OC_PLUGIN_DEBUG"]) {
						error_log("mapi exception: " . $e->getMessage());
					}
				} catch (AuthException $e) {
					//TODO: rewrite this dirty code...
					
					// always clear cache - otherwhise the next request will probably not fail...
					$this->cache->clear();
					
					if ($e->getCode() == 401) {
						$nodes["/"] = array(
							'id'	=>	"/",
							'path'	=>	"/",
							'text'	=>	$e->getCode() . " Username or Password wrong!",
							'filename'=> "error"
						);						
					} else if ($e->getCode() == 403) {
						$nodes["/"] = array(
							'id'	=>	"/",
							'path'	=>	"/",
							'text'	=>	$e->getCode() . " Could not connect to Server! Check your config!",
							'filename'=> "error"
						);
					} else if ($e->getCode() == 404) {
						$nodes["/"] = array(
							'id'	=>	"/",
							'path'	=>	"/",
							'text'	=>	$e->getCode() . " Unknown Owncloud Path or Server! Check your config!",
							'filename'=> "error"
						);
					}
					
					$response['items'] = array_values($nodes);
					$response['status']	=	false;
					$this->addActionData($actionType, $response);
					$GLOBALS["bus"]->addData($this->getResponseData());
					
					if($_ENV["OC_PLUGIN_DEBUG"]) {
						error_log("auth exception: " . $e->getMessage());
					}
				}
			}
		}
		return $result;
	}

	/**
	 * loads content of current folder - list of folders and files from Owncloud
	 * @param {Array} $actionType
	 * @param {Array} $actionData
	 */
	public function loadNode($actionType, $actionData) {	
		$nodeId = $actionData['id'];
		
		$cache_object = $this->cache->get($nodeId);
		if(!is_null($cache_object)) {
			$dir = $cache_object;
		} else {
			if (!$this->wdc->open()) {
				if($_ENV["OC_PLUGIN_DEBUG"]) {
					error_log('Error: could not open server connection');
				}
				throw new AuthException("403 Failed to connect",403);
			}
			
			/* logon to owncloud */
			$this->authorise();
			
			$dir = $this->wdc->ls($nodeId);
			$this->cache->put($nodeId, $dir);
				
			$this->wdc->close();
		}
		
		if($_ENV["OC_PLUGIN_DEBUG"]) {
			error_log("parsing dir: " . $dir);
		}
		if($dir == -401) {	// unauthorised!
			throw new AuthException("401 unauthorised",401);
		} else if($dir == -404) {	// unknown/not found!
			throw new AuthException("404 not found",404);
		}else {
			$response["item"] = Helper::parseFolderContent($nodeId, $dir, $actionData, false, false);
		}
		if($_ENV["OC_PLUGIN_DEBUG"]) {
			error_log("parsing done!");
		}
		
		$response['page'] = array("start" => 0, "rowcount" => 50, "totalrowcount" => count($response["item"]));
		$response['folder'] = array("content_count" => count($response["item"]), "content_unread" => 0);
		$this->addActionData($actionType, $response);
		$GLOBALS["bus"]->addData($this->getResponseData());
		
		if($_ENV["OC_PLUGIN_DEBUG"]) {
			error_log("nodes loaded, bus data written!");
		}
	}
	
	/**
	 * Deletes the selected files on the webdav server
	 * @param {Array} $actionType
	 * @param {Array} $actionData
	 * @private
	 */
	private function delete($actionType, $actionData) {
		if (!$this->wdc->open()) {
			if($_ENV["OC_PLUGIN_DEBUG"]) {
				error_log('Error: could not open server connection');
			}
			throw new AuthException("403 Failed to connect",403);
		}
		
		/* logon to owncloud */
		$this->authorise();
		
		$nodeId = $actionData['entryid'];
		$http_status = $this->wdc->delete($nodeId);
		
		$this->cache->remove($nodeId); // remove cached object...
		
		$this->wdc->close();
		
		if($_ENV["OC_PLUGIN_DEBUG"]) {
			error_log("delete status: " . $http_status["status"]);
		}
		
		if(intval($http_status["status"]) > 210) {	// something went wrong...
			$msg = "Deleting item failed! (" . $http_status['status'] . ")";
			$this->sendFeedback(false, array('error' => array(
				'type' => ERROR_GENERAL,
				'info' => array(
					'original_message' => $msg,
					'display_message' => $msg
				))
			));
		} else {		
			$response['status'] = true;
			$this->addActionData($actionType, $response);
			$GLOBALS["bus"]->addData($this->getResponseData());
		}
	}
	
	/**
	 * Handle open requests
	 * @param {Array} $actionType
	 * @param {Array} $actionData
	 * @private
	 */
	private function open($actionType, $actionData) {
		if (!$this->wdc->open()) {
			if($_ENV["OC_PLUGIN_DEBUG"]) {
				error_log('Error: could not open server connection');
			}
			throw new AuthException("403 Failed to connect",403);
		}
		
		/* logon to owncloud */
		$this->authorise();
		
		$nodeId = $actionData['entryid'];
		
		$node = $this->wdc->gpi($nodeId);
		$this->wdc->close();
		
		$filename =  basename($nodeId);
		$type = 1;
		if (strcmp($node['resourcetype'],"collection") == 0) { // we have a folder
			$type = 0;
		}
		
		error_log(print_r($node,true));
		
		$node = array('props' =>
			array(
				'id'	=>	urldecode($nodeId),
				'path'	=>	dirname(urldecode($nodeId)),
				'filename'=>$filename,							
				'message_size'	=>	$node['getcontentlength'] == null ? 0 : intval($node['getcontentlength']),
				'lastmodified' => strtotime($node['lastmodified']) * 1000,
				'message_class' => "IPM.Owncloud",
				'type'=> $type
			),
			'entryid' => urldecode($nodeId),
			'store_entryid' => 'owncloud',
			'parent_entryid' => dirname($nodeId) . "/"
		);
		$response['item'] = $node;
		$this->addActionData("item", $response);
		$GLOBALS["bus"]->addData($this->getResponseData());
	}
	
	/**
	 * create a new Directory
	 * @param {Array} $actionType
	 * @param {Array} $actionData
	 * @private
	 */
	private function createDirectory($actionType, $actionData) {
		if (!$this->wdc->open()) {
			if($_ENV["OC_PLUGIN_DEBUG"]) {
				error_log('Error: could not open server connection');
			}
			throw new AuthException("403 Failed to connect",403);
		}
		
		/* logon to owncloud */
		$this->authorise();
		
		$actionData = $actionData["props"];	
		$dirname = $actionData["id"];
		$http_status  = $this->wdc->mkcol($dirname); // create it !
		
		$parentdir =  dirname($dirname);  // get parent dir
		if($parentdir != "/") {
			$parentdir = $parentdir . "/";
		}
		
		$this->cache->remove($dirname);
		$this->cache->remove($parentdir);
		
		if($_ENV["OC_PLUGIN_DEBUG"]) {
			error_log("Created dir: " . $dirname . " response: " . $http_status);
		}
		
		$response = array();
		
		if(intval($http_status) > 210) {	// something went wrong...
			$msg = "Creating folder failed! (" . $http_status . ")";
			$this->sendFeedback(false, array('error' => array(
				'type' => ERROR_GENERAL,
				'info' => array(
					'original_message' => $msg,
					'display_message' => $msg
				))
			));
		} else {
			/* create the response folder object */
			$folder = array();
			
			$folder[$dirname] = array(
				'props' =>
					array(
						'id'	=>	urldecode($dirname),
						'path'	=>	dirname(urldecode($dirname)),
						'filename'=> $actionData["filename"],							
						'message_size'	=>	0,
						'lastmodified' => $actionData['lastmodified'],
						'message_class' => "IPM.Owncloud",
						'type'=> $actionData['type']
					),
				'entryid' => urldecode($dirname),
				'store_entryid' => 'owncloud',
				'parent_entryid' => $parentdir
			);
			$response['item'] = array_values($folder);
			
			$this->addActionData("update", $response);
			$GLOBALS["bus"]->addData($this->getResponseData());
		}
	}

	/**
	 * Sets access token for Owncloud's WebDav Interface
	 * @param {Array} $data
	 * @private
	 */
	private function authorise() {
		$this->wdc->set_user($GLOBALS["settings"]->get("zarafa/v1/contexts/owncloud/username"));
		$this->wdc->set_pass(base64_decode($GLOBALS["settings"]->get("zarafa/v1/contexts/owncloud/password")));
		
		if($_ENV["OC_PLUGIN_DEBUG"]) {
			error_log("auth for " . $GLOBALS["settings"]->get("zarafa/v1/contexts/owncloud/username"));
		}
	}
	/**
	 * Function will be used to process private items in a list response,
	 * owncloud doesn't have private items so function is overriden to not do 
	 * any processing.
	 * @param {Object} $item item properties
	 * @return {Object} item properties if its non private item otherwise empty array
	 */
	function processPrivateItem($item) {
		return $item;
	}
}
?>
