<?php
/**
 * module.owncloud.php, owncloud zarafa integration module
 *
 * Author: Christoph Haas <christoph.h@sprinternet.at>
 * Copyright (C) 2012-2013 Christoph Haas
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
 
include('version.php');
include('Owncloud/class.helper.php');
require_once('Owncloud/webdav.php');
require_once('Owncloud/authexception.php');
require_once('Owncloud/cachemanager.php');

/**
 * This module integrates Owncloud into attachment part of emails
 * @class OwncloudModule
 * @extends Module
 */
class OwncloudModule extends Module {

	private $wdc;				// webdav client
	private $cache;				// cache manager
	private $initdone = false;

	/**
	 * @constructor
	 * @param $id
	 * @param $data
	 */
	public function __construct($id, $data) {
			parent::Module($id, $data);
	}

	/**
	 * Initialize PHP API of Owncloud
	 * @param $data
	 */
	private function bootstrap($data) {
		$this->wdc = new webdav_client();
		$oc_base = $GLOBALS["settings"]->get("zarafa/v1/contexts/owncloud/owncloud_path");
		$oc_server = $GLOBALS["settings"]->get("zarafa/v1/contexts/owncloud/server");
		$oc_ssl = $GLOBALS["settings"]->get("zarafa/v1/contexts/owncloud/use_ssl");
		
		$this->wdc->set_server($oc_server);
		if($oc_ssl == true) {
			$this->wdc->set_port(intval($GLOBALS["settings"]->get("zarafa/v1/contexts/owncloud/port_ssl")));
		} else {
			$this->wdc->set_port(intval($GLOBALS["settings"]->get("zarafa/v1/contexts/owncloud/port")));
		}
		$this->wdc->set_ssl($oc_ssl);
		$this->wdc->set_pathprefix($oc_base);
		// use HTTP/1.1
		$this->wdc->set_protocol(1);
		// enable debugging
		$this->wdc->set_debug($_ENV["OC_PLUGIN_DEBUG"]);
		
		// initialize the cachemanager
		$this->cache = new cache_manager(PLUGIN_OWNCLOUDBROWSER_TMP, $GLOBALS["settings"]->get("zarafa/v1/contexts/owncloud/enable_caching"));
	}

	/**
	 * Executes all the actions in the $data variable.
	 * Exception part is used for authentication errors also
	 * @return boolean true on success or false on failure.
	 */
	public function execute() {
		$result = false;

		foreach($this->data as $actionType => $actionData)
		{			
			/* we have to fix this.... */
			//TODO: rewrite this dirty code...
			if(!$this->initdone && isset($actionData)) {
				$this->bootstrap($actionData);
				$this->initdone = true;
			}
			if(isset($actionType)) {
				try {
					if($_ENV["OC_PLUGIN_DEBUG"]) {
						error_log("exec: " . $actionType);
					}
					switch($actionType)
					{
						case "loaddirectory":
							$result = $this->loadNode($actionType, $actionData, !$actionData['loadfiles']);
							break;
						case "getversion":
							$result = $this->getVersion($actionData);
							break;
						case "download-to-tmp":
							$result = $this->downloadSelectedOwncloudFilesToTmp($actionData);
							break;
						case "prepare-attachment":
							$result = $this->prepareAttachmentToStore($actionData);
							break;
						case "createdir":
							$result = $this->createDirectory($actionData);
							break;
						case "uploadtooc":
							$result = $this->uploadToOwncloud($actionData);
							break;
						case "clearcache":
							$result = $this->clearCache($actionType, $actionData);
							break;
						default:
							$this->handleUnknownActionType($actionType);
					}

				} catch (MAPIException $e) {
					if($_ENV["OC_PLUGIN_DEBUG"]) {
						error_log("mapi exception: " . $e->getMessage());
					}
				} catch (AuthException $e) {
					
					// always clear cache - otherwhise the next request will probably not fail...
					$this->cache->clear();
					
					if ($e->getCode() == 401) {
						$message = $e->getCode() . " Username or password wrong!";		
					} else if ($e->getCode() == 403) {
						$message = $e->getCode() . " Could not connect to server! Check your config!";
					} else if ($e->getCode() == 404) {
						$message = $e->getCode() . " Unknown owncloud/webdav path or server! Check your config!";
					} else {
						$message = $e->getCode() . " WebDAV response!";
					}
					
					$response['error'] = $message;
					$response['status']	=	false;
					$this->addActionData($actionType, $response);
					$GLOBALS["bus"]->addData($this->getResponseData());
					
					if($_ENV["OC_PLUGIN_DEBUG"]) {
						error_log("auth exception: " . $e->getMessage());
					}
				}
			}
		}
		return $result;
	}
	
	/**
	 * clears the cache!
	 * @param {Array} $actionType
	 * @param {Array} $actionData
	 */
	public function clearCache($actionType, $actionData) {
		$this->cache->clear();
		
		$response['status']	=	true;
		$this->addActionData($actionType, $response);
		$GLOBALS["bus"]->addData($this->getResponseData());
	}

	/**
	 * loads content of current folder - list of folders and files from Owncloud
	 * @param {Array} $actionData
	 */
	public function loadNode($actionType, $actionData, $onlyfolders = false) {
	
		$nodeId = $actionData['id'];
		
		$cache_object = $this->cache->get($nodeId);
		if(!is_null($cache_object)) {
			$dir = $cache_object;
		} else {		
			if (!$this->wdc->open()) {
				if($_ENV["OC_PLUGIN_DEBUG"]) {
					error_log('Error: could not open server connection');
				}
				throw new AuthException("403 Failed to connect",403);
			}
					
			/* logon to owncloud */
			$this->authorise();
			
			$dir = $this->wdc->ls($nodeId);
			$this->cache->put($nodeId, $dir);
			
			$this->wdc->close();
		}
		
		if($_ENV["OC_PLUGIN_DEBUG"]) {
			error_log("parsing dir: " . $dir);
		}
		if($dir == -401) {	// unauthorised!
			throw new AuthException("401 unauthorised",401);
		} else if($dir == -404) {	// unknown/not found!
			throw new AuthException("404 not found",404);
		}else {
			$response["items"] = Helper::parseFolderContent($nodeId, $dir, $actionData, $onlyfolders, true);
		}
		
		$response['status']	=	true;
		$this->addActionData($actionType, $response);
		$GLOBALS["bus"]->addData($this->getResponseData());
		
		if($_ENV["OC_PLUGIN_DEBUG"]) {
			error_log("nodes loaded, bus data written!");
		}
	}

	/**
	 * Downloads file from the Owncloud service and saves it in tmp
	 * folder with unique name
	 * @param {Array} $actionData
	 * @private
	 */
	private function downloadSelectedOwncloudFilesToTmp($actionData) {
		$ids = $actionData['ids'];
		$dialogAttachmentId = $actionData['dialog_attachments'];
		$oc_base = $GLOBALS["settings"]->get("zarafa/v1/contexts/owncloud/owncloud_path");
		$response = array();
		
		if (!$this->wdc->open()) {
			if($_ENV["OC_PLUGIN_DEBUG"]) {
				error_log('Error: could not open server connection');
			}
			throw new AuthException("403 Failed to connect",403);
		}
		
		/* logon to owncloud */
		$this->authorise();
		
		$attachment_state = new AttachmentState();
		
		// WORKAROUND TO KEEP COMPATIBILITY FOR OLDER WEBAPPS (<=1.2.1)
		if(method_exists($attachment_state,"open")) {
			$attachment_state->open();
		}
		
		foreach ($ids as $id) {
			$file = $id;
			$filename = basename($file);
			
			// WORKAROUND TO KEEP COMPATIBILITY FOR OLDER WEBAPPS (<=1.2.1)
			if(method_exists($attachment_state,"getAttachmentTmpPath")) {
				$tmpname = $attachment_state->getAttachmentTmpPath($filename);
			} else {
				$tmpname = tempnam(TMP_PATH, stripslashes($filename));
			}
			
			$path = dirname($file);
			
			if($_ENV["OC_PLUGIN_DEBUG"]) {
				error_log("Downloading: " . $filename . " to: " . $tmpname);
			}
			/* download file from webdav */
			$buffer = null;
			$http_status = $this->wdc->get($file, $buffer);
			if($_ENV["OC_PLUGIN_DEBUG"]) {
				error_log("http-status: " . $http_status);
			}
			
			$fhandle = fopen($tmpname,'w');
			fwrite($fhandle,$buffer,strlen($buffer));
			fclose($fhandle); 
			
			$response['items'][] = array(
				'name' => $filename,
				'size' => filesize($tmpname),
				'tmpname'=>Helper::getFilenameFromPath($tmpname)
			);
			
			// mimetype is not required...
			$attachment_state->addAttachmentFile($dialogAttachmentId, Helper::getFilenameFromPath($tmpname), Array(
				"name"       => $filename,
				"size"       => filesize($tmpname),
				"sourcetype" => 'default'
			));
			if($_ENV["OC_PLUGIN_DEBUG"]) {
				error_log("filesize: " . filesize($tmpname));
			}
		}
		
		$this->wdc->close();
		$attachment_state->close();
		$response['status'] = true;
		$this->addActionData("tmpdownload", $response);
		$GLOBALS["bus"]->addData($this->getResponseData());
	}

	/**
	 * Sets access token for Owncloud's WebDav Interface
	 * @param {Array} $data
	 * @private
	 */
	private function authorise() {
		$this->wdc->set_user($GLOBALS["settings"]->get("zarafa/v1/contexts/owncloud/username"));
		$this->wdc->set_pass(base64_decode($GLOBALS["settings"]->get("zarafa/v1/contexts/owncloud/password")));
		
		if($_ENV["OC_PLUGIN_DEBUG"]) {
			error_log("auth for " . $GLOBALS["settings"]->get("zarafa/v1/contexts/owncloud/username"));
		}
	}
	
	/**
	 * upload the tempfile to owncloud
	 * @param {Array} $data
	 * @private
	 */
	private function uploadToOwncloud($data) {
	
		if (!$this->wdc->open()) {
			if($_ENV["OC_PLUGIN_DEBUG"]) {
				error_log('Error: could not open server connection');
			}
			throw new AuthException("403 Failed to connect",403);
		}
		
		/* logon to owncloud */
		$this->authorise();
		
		$filename = $data["tmpfile"];
		$handle = fopen ($filename, 'r');
		$contents = fread ($handle, filesize ($filename));
		fclose ($handle);
		$target_path = $data["destdir"] . $data["destname"];		
		$http_status = $this->wdc->put($target_path,$contents);
		
		if($_ENV["OC_PLUGIN_DEBUG"]) {
			error_log("Uploaded to: " . $target_path . " response: " . $http_status);
		}
		
		$response = array();
		$response['status'] = $http_status;
		$this->addActionData("uploadtooc", $response);
		$GLOBALS["bus"]->addData($this->getResponseData());
	}
	
	/**
	 * return http response code
	 * @param {String} $url to check
	 * @return {String} response
	 * @private
	 */
	private function get_http_response_code($url) {
		$headers = get_headers($url);
		return substr($headers[0], 9, 3);
	}
	
	/**
	 * return owncloud version
	 * @param {Array} $data
	 * @private
	 */
	private function getVersion($data) {
		$oc_base = $GLOBALS["settings"]->get("zarafa/v1/contexts/owncloud/owncloud_path");
		$oc_server = $GLOBALS["settings"]->get("zarafa/v1/contexts/owncloud/server");
		$response = array();
		$response['ocversion'] = "Unknown";
		
		if(!empty($oc_server) && $oc_server != "") {
			$url = "http://" . $oc_server . substr($oc_base, 0, strlen($oc_base) - strlen("files/remote.php")) . "status.php"; // old url
			$url2 = "http://" . $oc_server . substr($oc_base, 0, strlen($oc_base) - strlen("remote.php/webdav")) . "status.php"; // new url
			
			if($this->get_http_response_code($url) == "200"){
				$status = file_get_contents($url);
				$versions = json_decode($status);
				$response['ocversion'] = $versions->versionstring;
			} else if($this->get_http_response_code($url2) == "200"){
				$status = file_get_contents($url2);
				$versions = json_decode($status);
				$response['ocversion'] = $versions->versionstring;
			} else {
				$response['ocversion'] = "Undetected (no Owncloud?)";
			}
		}		
		
		$response['version'] = $_ENV["OC_PLUGIN_VERSION"];
		$this->addActionData("getversion", $response);
		$GLOBALS["bus"]->addData($this->getResponseData());
	}
	
	/**
	 * create a new Directory
	 * @param {Array} $data
	 * @private
	 */
	private function createDirectory($data) {
	
		if (!$this->wdc->open()) {
			if($_ENV["OC_PLUGIN_DEBUG"]) {
				error_log('Error: could not open server connection');
			}
			throw new AuthException("403 Failed to connect",403);
		}
		
		$oc_base = $GLOBALS["settings"]->get("zarafa/v1/contexts/owncloud/owncloud_path");
		
		/* logon to owncloud */
		$this->authorise();
		
		$dirname = $data["dirname"];
		$http_status  = $this->wdc->mkcol($dirname);
		
		if($_ENV["OC_PLUGIN_DEBUG"]) {
			error_log("Created dir: " . $dirname . " response: " . $http_status);
		}
		
		$response = array();
		$response['status'] = $http_status;
		$response['basedir'] = $data['basedir'];
		$this->addActionData("createdir", $response);
		$GLOBALS["bus"]->addData($this->getResponseData());
	}
	
	/**
	 * Store the file to a temporary directory, prepare it for oc upload
	 * @param {Array} $data
	 * @private
	 */
	private function prepareAttachmentToStore($data) {
		
		if($_ENV["OC_PLUGIN_DEBUG"]) {
			error_log("preparing attachment");
		}
	
		// Get store id
		$storeid = false;
		if(isset($data["store"])) {
			$storeid = $data["store"];
		}

		// Get message entryid
		$entryid = false;
		if(isset($data["entryid"])) {
			$entryid = $data["entryid"];
		}

		// Check which type isset
		$openType = "attachment";

		// Get number of attachment which should be opened.
		$attachNum = false;
		if(isset($data["attachNum"])) {
			$attachNum = $data["attachNum"];
		}

		// Check if storeid and entryid isset
		if($storeid && $entryid) {
			// Open the store
			$store = $GLOBALS["mapisession"]->openMessageStore(hex2bin($storeid));
			
			if($store) {
				// Open the message
				$message = mapi_msgstore_openentry($store, hex2bin($entryid));
				
				if($message) {
					$attachment = false;

					// Check if attachNum isset
					if($attachNum) {
						// Loop through the attachNums, message in message in message ...
						for($i = 0; $i < (count($attachNum) - 1); $i++)
						{
							// Open the attachment
							$tempattach = mapi_message_openattach($message, (int) $attachNum[$i]);
							if($tempattach) {
								// Open the object in the attachment
								$message = mapi_attach_openobj($tempattach);
							}
						}

						// Open the attachment
						$attachment = mapi_message_openattach($message, (int) $attachNum[(count($attachNum) - 1)]);
					}

					// Check if the attachment is opened
					if($attachment) {
						
						// Get the props of the attachment
						$props = mapi_attach_getprops($attachment, array(PR_ATTACH_LONG_FILENAME, PR_ATTACH_MIME_TAG, PR_DISPLAY_NAME, PR_ATTACH_METHOD));
						// Content Type
						$contentType = "application/octet-stream";
						// Filename
						$filename = "ERROR";

						// Set filename
						if(isset($props[PR_ATTACH_LONG_FILENAME])) {
							$filename = $props[PR_ATTACH_LONG_FILENAME];
						} else if(isset($props[PR_ATTACH_FILENAME])) {
							$filename = $props[PR_ATTACH_FILENAME];
						} else if(isset($props[PR_DISPLAY_NAME])) {
							$filename = $props[PR_DISPLAY_NAME];
						} 
				
						// Set content type
						if(isset($props[PR_ATTACH_MIME_TAG])) {
							$contentType = $props[PR_ATTACH_MIME_TAG];
						} else {
							// Parse the extension of the filename to get the content type
							if(strrpos($filename, ".") !== false) {
								$extension = strtolower(substr($filename, strrpos($filename, ".")));
								$contentType = "application/octet-stream";
								if (is_readable("mimetypes.dat")){
									$fh = fopen("mimetypes.dat","r");
									$ext_found = false;
									while (!feof($fh) && !$ext_found){
										$line = fgets($fh);
										preg_match("/(\.[a-z0-9]+)[ \t]+([^ \t\n\r]*)/i", $line, $result);
										if ($extension == $result[1]){
											$ext_found = true;
											$contentType = $result[2];
										}
									}
									fclose($fh);
								}
							}
						}
						
						
						$tmpname = tempnam(TMP_PATH, stripslashes($filename));

						// Open a stream to get the attachment data
						$stream = mapi_openpropertytostream($attachment, PR_ATTACH_DATA_BIN);
						$stat = mapi_stream_stat($stream);
						// File length =  $stat["cb"]
						
						if($_ENV["OC_PLUGIN_DEBUG"]) {
							error_log("filesize: " . $stat["cb"]);
						}
						
						$fhandle = fopen($tmpname,'w');
						$buffer = null;
						for($i = 0; $i < $stat["cb"]; $i += BLOCK_SIZE) {
							// Write stream						
							$buffer = mapi_stream_read($stream, BLOCK_SIZE);
							fwrite($fhandle,$buffer,strlen($buffer));
						}
						fclose($fhandle);
						
						if($_ENV["OC_PLUGIN_DEBUG"]) {
							error_log("temp attachment written to " . $tmpname);
						}
						
						$response = array();
						$response['tmpname'] = $tmpname;
						$response['filename'] = $filename;
						$response['status'] = true;
						$this->addActionData("prepareattachment", $response);
						$GLOBALS["bus"]->addData($this->getResponseData());
					}
				}
			} else {
				if($_ENV["OC_PLUGIN_DEBUG"]) {
					error_log("store could not be opened");
				}
			}
		} else {
			if($_ENV["OC_PLUGIN_DEBUG"]) {
				error_log("wrong call, store and entryid have to be set");
			}
		}
	}
};

?>
