<?php
/**
 * classowncloudbrowsermodule.php, owncloud zarafa integration module
 *
 * Author: Christoph Haas <christoph.h@sprinternet.at>
 * Copyright (C) 2012-2013 Christoph Haas
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
 
include('version.php');
include('Owncloud/class.helper.php');
require_once('Owncloud/webdav.php');
require_once('Owncloud/webdavexception.php');
require_once('Owncloud/cachemanager.php');

/**
 * This module handles all list and change requests for the webdav browser.
 *
 * @class OwncloudBrowserModule
 * @extends ListModule
 */
class OwncloudBrowserModule extends ListModule {

	/**
	 * @var object The webdav-client object
	 */
	private $wdc;
	
	/**
	 * @var object The cachemanager object
	 */
	private $cache;

	/**
	 * @constructor
	 * @param int $id unique id.
	 * @param array $data list of all actions.
	 */
	function OwncloudBrowserModule($id, $data) {
		parent::ListModule($id, $data);

		$this->start = 0;
	}

	/**
	 * Initialize PHP API of Owncloud
	 *
	 * @return void
	 */
	private function bootstrap() {
		$this->wdc = new webdav_client();
		$oc_base = $GLOBALS["settings"]->get("zarafa/v1/contexts/owncloud/owncloud_path");
		$oc_server = $GLOBALS["settings"]->get("zarafa/v1/contexts/owncloud/server");
		$oc_ssl = $GLOBALS["settings"]->get("zarafa/v1/contexts/owncloud/use_ssl");
		
		$this->wdc->set_server($oc_server);
		if($oc_ssl == true) {
			$this->wdc->set_port(intval($GLOBALS["settings"]->get("zarafa/v1/contexts/owncloud/port_ssl")));
		} else {
			$this->wdc->set_port(intval($GLOBALS["settings"]->get("zarafa/v1/contexts/owncloud/port")));
		}
		$this->wdc->set_ssl($oc_ssl);
		$this->wdc->set_pathprefix($oc_base);
		// use HTTP/1.1
		$this->wdc->set_protocol(1);
		// enable debugging
		$this->wdc->set_debug($_ENV["OC_PLUGIN_DEBUG"]);
		
		// initialize the cachemanager
		$this->cache = new cache_manager(PLUGIN_OWNCLOUDBROWSER_TMP, $GLOBALS["settings"]->get("zarafa/v1/contexts/owncloud/enable_caching"));
	}

	/**
	 * Executes all the actions in the $data variable.
	 * Exception part is used for webdav errors also.
	 *
	 * @return boolean true on success or false on failure.
	 */
	public function execute() {
		$result = false;
		
		$this->bootstrap();
		
		foreach($this->data as $actionType => $actionData) {
			if(isset($actionType)) {
				try {
					Helper::log("exec: " . $actionType, "Browser");
					
					switch($actionType) {
						case "list":
							$result = $this->loadNode($actionType, $actionData);
							break;
						case "delete":
							$result = $this->delete($actionType, $actionData);
							break;
						case "open":
							$result = $this->open($actionType, $actionData);
							break;
						case "move":
							$result = $this->move($actionType, $actionData);
							break;
						case "save":
							$result = $this->save($actionType, $actionData);
							break;
						default:
							$this->handleUnknownActionType($actionType);
					}

				} catch (MAPIException $e) {
					Helper::log("mapi exception: " . $e->getMessage(), "Browser");
				} catch (WebDAVException $e) {
					// always clear cache - otherwhise the next request will probably not fail...
					$this->cache->clear();
					$response['items'] = array(); // respond nothing, errors will be handled by owncloudmodule
					$response['status'] = false;
					$this->addActionData($actionType, $response);
					$GLOBALS["bus"]->addData($this->getResponseData());
					
					Helper::log("auth exception: " . $e->getMessage(), "Browser");
				}
			}
		}
		return $result;
	}

	/**
	 * loads content of current folder - list of folders and files from Owncloud
	 *
	 * @param array $actionType
	 * @param array $actionData
	 * @throws WebDAVException if the webdav request fails
	 * @return void
	 */
	public function loadNode($actionType, $actionData) {
		$nodeId = $actionData['id'];
		$cache_object = $this->cache->get($nodeId);
		
		if(!is_null($cache_object)) {
			$dir = $cache_object;
		} else {
			if (!$this->wdc->open()) {
				Helper::log('Error: could not open server connection', "Browser");
				
				throw new WebDAVException("403 Failed to connect",403);
			}
			
			/* logon to owncloud */
			$this->authorise();
			
			$dir = $this->wdc->ls($nodeId);
			$this->cache->put($nodeId, $dir);
			
			$this->wdc->close();
		}
		
		Helper::log("parsing dir: " . $dir, "Browser");
		
		if($dir == -401) { // unauthorised!
			throw new WebDAVException("401 unauthorised",401);
		} else if($dir == -404) { // unknown/not found!
			throw new WebDAVException("404 not found",404);
		}else {
			$response["item"] = Helper::parseFolderContent($nodeId, $dir, $actionData, false, false);
		}
		Helper::log("parsing done!", "Browser");
		
		$response['page'] = array("start" => 0, "rowcount" => 50, "totalrowcount" => count($response["item"]));
		$response['folder'] = array("content_count" => count($response["item"]), "content_unread" => 0);
		$this->addActionData($actionType, $response);
		$GLOBALS["bus"]->addData($this->getResponseData());
		
		Helper::log("nodes loaded, bus data written!", "Browser");
	}
	
	/**
	 * Deletes the selected files on the webdav server
	 *
	 * @param array $actionType
	 * @param array $actionData
	 * @throws WebDAVException if the webdav request fails
	 * @return void
	 * @access private
	 */
	private function delete($actionType, $actionData) {
		if (!$this->wdc->open()) {
			Helper::log('Error: could not open server connection', "Browser");
			
			throw new WebDAVException("403 Failed to connect",403);
		}
		
		/* logon to owncloud */
		$this->authorise();
		
		$nodeId = $actionData['entryid'];
		$http_status = $this->wdc->delete($nodeId);
		Helper::log("deleted: " . $nodeId . " with status: " . $http_status, "Browser");
		
		$cacheKey = rtrim($nodeId, "/"); // this just hits folders...
		$this->cache->remove($cacheKey); // remove cached object...
		
		$this->wdc->close();
		
		if(intval($http_status["status"]) > 210) { // something went wrong...
			$msg = "Deleting item failed! (" . $http_status['status'] . ")";
			$this->sendFeedback(false, array(
				'type' => ERROR_GENERAL,
				'info' => array(
					'original_message' => $msg,
					'display_message' => $msg
				)
			));
		} else {
			$response['status'] = true;
			$this->addActionData($actionType, $response);
			$GLOBALS["bus"]->addData($this->getResponseData());
		}
	}
	
	/**
	 * Handle open requests
	 *
	 * @param array $actionType
	 * @param array $actionData
	 * @throws WebDAVException if the webdav request fails
	 * @return void
	 * @access private
	 */
	private function open($actionType, $actionData) {
		if (!$this->wdc->open()) {
			Helper::log('Error: could not open server connection', "Browser");
			
			throw new WebDAVException("403 Failed to connect",403);
		}
		
		/* logon to owncloud */
		$this->authorise();
		
		$nodeId = $actionData['entryid'];
		
		$node = $this->wdc->gpi($nodeId);
		$this->wdc->close();
		
		$filename =  basename($nodeId);
		$type = 1;
		if (strcmp($node['resourcetype'],"collection") == 0) { // we have a folder
			$type = 0;
		}
		
		$node = array('props' =>
			array(
				'id' => urldecode($nodeId),
				'path' => dirname(urldecode($nodeId)),
				'filename'=>$filename,
				'message_size' => $node['getcontentlength'] == null ? 0 : intval($node['getcontentlength']),
				'lastmodified' => strtotime($node['lastmodified']) * 1000,
				'message_class' => "IPM.Owncloud",
				'type'=> $type
			),
			'entryid' => urldecode($nodeId),
			'store_entryid' => 'owncloud',
			'parent_entryid' => dirname($nodeId) . "/"
		);
		$response['item'] = $node;
		$this->addActionData("item", $response);
		$GLOBALS["bus"]->addData($this->getResponseData());
	}
	
	/**
	 * Handles the saverequests. Those requests could either be a new folder creation or a renamed file
	 *
	 * @param array $actionType
	 * @param array $actionData
	 * @return void
	 * @access private
	 */
	private function save($actionType, $actionData) {
		if(count($actionData["props"]) <= 2) { // we only submit the new filename and id on a rename...
			$this->rename($actionType, $actionData);
		} else {
			$this->createDirectory($actionType, $actionData);
		}
	}
	
	/**
	 * Moves the selected files on the webdav server
	 *
	 * @param array $actionType
	 * @param array $actionData
	 * @throws WebDAVException if the webdav request fails
	 * @return void
	 * @access private
	 */
	private function move($actionType, $actionData) {
		if (!$this->wdc->open()) {
			Helper::log('Error: could not open server connection', "Browser");
			
			throw new WebDAVException("403 Failed to connect",403);
		}
		
		/* logon to owncloud */
		$this->authorise();
		
		$dst = rtrim($actionData['destination'], '/');
		$overwrite = false;
		if(isset($actionData['overwrite']) && $actionData['overwrite'] === true) {
			$overwrite = true;
		}
		$http_status = array();
		$overall_status = true;
		$message = "";
		$errorids = array();
		
		for($i = 0; $i < count($actionData['ids']); $i++) {
			$source = rtrim($actionData['ids'][$i], '/');			
			$destination = $dst . '/' . Helper::getFilenameFromPath($source);
			
			$http_status[$i] = $this->wdc->move($source, $destination, $overwrite);
			$this->cache->remove($source);
			
			if(intval($http_status[$i]) > 210) {
				$overall_status = false;
				$errorids[] = $actionData['ids'][$i];
				$message = "Moving item " . $actionData['ids'][$i] . " to " . $destination . " failed! (" . $http_status[$i] . ")";
			}
			
			Helper::log("move status: " . $http_status[$i], "Browser");
		}
		
		$this->cache->remove($actionData['destination']); // clear cache for destination folder too
		$this->wdc->close();
		
		if(!$overall_status) { // something went wrong...
			$response['faileditems'] = $errorids;
			$response['message'] = $message;
		}
		
		$response['status'] = $overall_status;
		$this->addActionData($actionType, $response);
		$GLOBALS["bus"]->addData($this->getResponseData());
	}
	
	/**
	 * Renames the selected file on the webdav server
	 *
	 * @param array $actionType
	 * @param array $actionData
	 * @throws WebDAVException if the webdav request fails
	 * @return void
	 * @access private
	 */
	private function rename($actionType, $actionData) {
		if (!$this->wdc->open()) {
			Helper::log('Error: could not open server connection', "Browser");
			
			throw new WebDAVException("403 Failed to connect",403);
		}
		
		/* logon to owncloud */
		$this->authorise();
		
		$src = rtrim($actionData['entryid'], '/');
		$dstdir = dirname($src) == "/" ? "" : dirname($src);
		$dst = $dstdir . "/" .  rtrim($actionData['props']['filename'], '/');
		
		$http_status = $this->wdc->move($src, $dst, false);
		$this->wdc->close();
		
		$this->cache->remove($src);
		
		Helper::log("Renamed: " . $src . " to: " . $dst . " with response: " . $http_status, "Browser");
			
		if(intval($http_status) > 210) {
			$msg = "Renaming failed! (" . $http_status . ")";
			$this->sendFeedback(false, array(
				'type' => ERROR_GENERAL,
				'info' => array(
					'original_message' => $msg,
					'display_message' => $msg
				)
			));
		} else {
			/* create the response object */
			$folder = array();
			
			$folder[dirname($src)] = array(
				'props' =>
					array(
						'id' => urldecode($dst),
						'path' => dirname(urldecode($dst)),
						'filename'=> $actionData['props']['filename']
					),
				'entryid' => urldecode($dst),
				'store_entryid' => 'owncloud',
				'parent_entryid' => dirname(dirname($src))
			);
			$response['item'] = array_values($folder);
			
			$this->addActionData("update", $response);
			$GLOBALS["bus"]->addData($this->getResponseData());
		}
	}
	
	/**
	 * Creates a new directory.
	 *
	 * @param array $actionType
	 * @param array $actionData
	 * @throws WebDAVException if the webdav request fails
	 * @return void
	 * @access private
	 */
	private function createDirectory($actionType, $actionData) {
		if (!$this->wdc->open()) {
			Helper::log('Error: could not open server connection', "Browser");
			
			throw new WebDAVException("403 Failed to connect",403);
		}
		
		/* logon to owncloud */
		$this->authorise();
		
		$actionData = $actionData["props"];
		$dirname = $actionData["id"];
		$http_status  = $this->wdc->mkcol($dirname); // create it !
		
		$parentdir =  dirname($dirname);  // get parent dir
		if($parentdir != "/") {
			$parentdir = $parentdir . "/";
		}
		
		$this->cache->remove($parentdir);
		
		Helper::log("Created dir: " . $dirname . " response: " . $http_status, "Browser");
		
		$response = array();
		
		if(intval($http_status) > 210) { // something went wrong...
			$msg = "Creating folder failed! (" . $http_status . ")";
			$this->sendFeedback(false, array(
				'type' => ERROR_GENERAL,
				'info' => array(
					'original_message' => $msg,
					'display_message' => $msg
				)
			));
		} else {
			/* create the response folder object */
			$folder = array();
			
			$folder[$dirname] = array(
				'props' =>
					array(
						'id' => urldecode($dirname),
						'path' => dirname(urldecode($dirname)),
						'filename'=> $actionData["filename"],
						'message_size' => 0,
						'lastmodified' => $actionData['lastmodified'],
						'message_class' => "IPM.Owncloud",
						'type'=> $actionData['type']
					),
				'entryid' => urldecode($dirname),
				'store_entryid' => 'owncloud',
				'parent_entryid' => $parentdir
			);
			$response['item'] = array_values($folder);
			
			$this->addActionData("update", $response);
			$GLOBALS["bus"]->addData($this->getResponseData());
		}
	}

	/**
	 * Sets access token for Owncloud's WebDav Interface
	 *
	 * @param array $data
	 * @return void
	 * @access private
	 */
	private function authorise() {
		$this->wdc->set_user($GLOBALS["settings"]->get("zarafa/v1/contexts/owncloud/username"));
		$this->wdc->set_pass(base64_decode($GLOBALS["settings"]->get("zarafa/v1/contexts/owncloud/password")));
		
		Helper::log("auth for " . $GLOBALS["settings"]->get("zarafa/v1/contexts/owncloud/username"), "Browser");
	}
	
	/**
	 * Function will be used to process private items in a list response,
	 * owncloud doesn't have private items so function is overriden to not do 
	 * any processing.
	 *
	 * @param object $item item properties
	 * @return object item properties if its non private item otherwise empty array
	 */
	function processPrivateItem($item) {
		return $item;
	}
}
?>
