#ifndef ECLOGGER_H
#define ECLOGGER_H

#include <string>
#include <pthread.h>
#include <stdio.h>
#include <stdarg.h>
#include "ECConfig.h"

#define EC_LOGLEVEL_NONE	0
#define EC_LOGLEVEL_FATAL	1
#define EC_LOGLEVEL_ERROR	2
#define EC_LOGLEVEL_WARNING	3
#define EC_LOGLEVEL_INFO	4
#define EC_LOGLEVEL_ANY		5
#define EC_LOGLEVEL_DEBUG	6

#define _LOG_BUFSIZE		10240
#define _LOG_TSSIZE			64

class ECLogger {
protected:
	char* MakeTimestamp();
	int max_loglevel;
	char *msgbuffer;
	char timestring[_LOG_TSSIZE];

public:
	ECLogger(int max_ll);
	virtual ~ECLogger();

	virtual void Reset() = 0;
	virtual void Log(int loglevel, const std::string& message) = 0;
	virtual void Log(int loglevel, char *format, ...) = 0;
};


/* Dummy null logger */
class ECLogger_Null : public ECLogger {
public:
	ECLogger_Null();
	~ECLogger_Null();

	virtual void Reset();
	virtual void Log(int loglevel, const std::string& message);
	virtual void Log(int loglevel, char *format, ...);
};

/* File logger. Use "-" for stderr logging */
class ECLogger_File : public ECLogger {
private:
	FILE *log;
	pthread_mutex_t filelock;
	char *logname;
	int timestamp;

public:
	ECLogger_File(int max_ll, int add_timestamp, char *filename);
	~ECLogger_File();

	virtual void Reset();
	virtual void Log(int loglevel, const std::string& message);
	virtual void Log(int loglevel, char *format, ...);
};

/*	Linux syslog logger */
#ifdef LINUX
class ECLogger_Syslog : public ECLogger {
private:
	int levelmap[EC_LOGLEVEL_DEBUG+1];	/* converts to syslog levels */

public:
	ECLogger_Syslog(int max_ll, char *ident);
	~ECLogger_Syslog();

	virtual void Reset();
	virtual void Log(int loglevel, const std::string& message);
	virtual void Log(int loglevel, char *format, ...);
};
#endif

/* Windows eventlog logger */
#if defined(_WIN32) && !defined(WINCE)
class ECLogger_Eventlog : public ECLogger {
private:
	WORD levelmap[EC_LOGLEVEL_DEBUG+1];	/* converts to eventlog levels */

	void ReportEventLog(WORD wType, const char* szMsg);
	HANDLE m_hEventSource;
	char m_szServiceName[64];

public:
	ECLogger_Eventlog(int max_ll, char* lpszServiceName);
	~ECLogger_Eventlog();

	virtual void Reset();
	virtual void Log(int loglevel, const std::string& message);
	virtual void Log(int loglevel, char *format, ...);
};
#endif

ECLogger* CreateLogger(ECConfig *config, char *argv0, char* lpszServiceName);
void LogConfigErrors(ECConfig* lpConfig, ECLogger* lpLogger);

#endif
