#include <errno.h>
#include <netinet/in.h>
#include <sys/mman.h>
#include <unistd.h>

#include "innoinfo.h"
#include "record.h"
#include "page.h"

#define USE_MMAP 0

unsigned int read_ulint(char *offset) {
  return ntohl(*(unsigned int *)offset);
}

unsigned long read_dulint(char *offset) {
  return ((unsigned long long)read_ulint(offset)) << 32 | read_ulint(offset + 4);
}

unsigned int read_ulshort(char *offset) {
  return ntohs(*(unsigned short int *)offset);
}

int read_lshort(char *offset) {
  return (short)ntohs(*(unsigned short int *)offset);
}

unsigned int read_ulchar(char *offset) {
  return *(unsigned char *)offset;
}

void dump_index_page_info(char *page) {
  printf(" Slots: %d\n", read_ulshort(page + PAGE_HEADER + PAGE_N_DIR_SLOTS));
  printf(" Heap top: %d\n", read_ulshort(page + PAGE_HEADER + PAGE_HEAP_TOP));
  printf(" Heap: %d (%s)\n", read_ulshort(page + PAGE_HEADER + PAGE_N_HEAP), read_ulshort(page + PAGE_HEADER + PAGE_N_HEAP) & 0x8000 ? "COMPACT" : "NORMAL");
  printf(" Free: %d\n", read_ulshort(page + PAGE_HEADER + PAGE_FREE));
  printf(" Garbage: %d\n", read_ulshort(page + PAGE_HEADER + PAGE_GARBAGE));
  printf(" Last insert: %d\n", read_ulshort(page + PAGE_HEADER + PAGE_LAST_INSERT));
  printf(" Direction: %d\n", read_ulshort(page + PAGE_HEADER + PAGE_DIRECTION));
  printf(" N Direction: %d\n", read_ulshort(page + PAGE_HEADER + PAGE_N_DIRECTION));
  printf(" Records: %d\n", read_ulshort(page + PAGE_HEADER + PAGE_N_RECS));
  printf(" Max Transaction ID: %ld\n", read_dulint(page + PAGE_HEADER + PAGE_MAX_TRX_ID));
  printf(" Page level: %d\n", read_ulshort(page + PAGE_HEADER + PAGE_LEVEL));
  printf(" Index ID: %lld\n", read_dulint(page + PAGE_HEADER + PAGE_INDEX_ID));
  fflush(stdout);
}


void dump_page_info(char *page, unsigned int pagenr) {
  printf("Page info for page %d:\n", pagenr);
  printf(" Next page: %d\n", read_ulint(page + FIL_PAGE_NEXT));
  printf(" Prev page: %d\n", read_ulint(page + FIL_PAGE_PREV));
  printf(" Page type: %d\n", read_ulshort(page + FIL_PAGE_TYPE));
  printf(" Space: %d\n", read_ulshort(page + FIL_PAGE_ARCH_LOG_NO_OR_SPACE_ID));
  if(read_ulshort(page + FIL_PAGE_TYPE) == 17855 ) {
    dump_index_page_info(page);
  }
  printf("\n");
  fflush(stdout);
}

char *page_read(int fd, unsigned long ulPage)
{
  char *ret;
#if USE_MMAP
  ret = (char *)mmap(NULL, PAGE_SIZE, PROT_READ, MAP_SHARED, fd, (off_t)ulPage << PAGE_SIZE_SHIFT);
  if(ret == MAP_FAILED) {
    fprintf(stderr,"mmap failed: %s\n", strerror(errno));
    return NULL;
  }
#else
  ret = (char *)malloc(PAGE_SIZE);
  if(pread(fd, ret, PAGE_SIZE, (off_t)ulPage << PAGE_SIZE_SHIFT) == -1) {
    fprintf(stderr,"pread failed: %s\n", strerror(errno));
    free(ret);
    return NULL;
  }
#endif

  if(ret && opt_verbose) {
    dump_page_info(ret, ulPage);
  }
  return ret;
}

void page_free(char *page) {
#if USE_MMAP
  munmap(page, PAGE_SIZE);
#else
  free(page);
#endif
}

int page_index_is_comp(char *pagedata) {
  int heap = read_ulshort(pagedata + PAGE_HEADER + PAGE_N_HEAP);
  if(heap & 0x8000)
    return 1;
  return 0;
}

int page_get_next(char *pagedata) {
  return read_ulint(pagedata + FIL_PAGE_NEXT);
}

int page_is_leaf(char *pagedata) {
  return read_ulshort(pagedata + PAGE_HEADER + PAGE_LEVEL) == 0;  
}

int page_get_leftmost(std::vector<INNOCOLUMN> columns, char *pagedata) {
  char *rec = rec_get_infimum(pagedata);
  int comp = page_index_is_comp(pagedata);

  rec = rec_get_next(comp, pagedata, rec);

  int fields = rec_get_field_count(columns, comp, rec, pagedata);
  return rec_get_nth_field_int(columns, comp, rec,fields-1, 1, pagedata); // last field is page
}

/*int page_get_rightmost(char *pagedata) {
  char *rec = rec_get_supremum(pagedata);
  rec = rec_get_prev(rec, pagedata);
  
  if(page_index_is_comp(pagedata))
    return NULL;
  else {
    int fields = rec_get_field_count_old(rec);
    return rec_get_nth_field_int_old(rec,fields-1); // last field is page
  }
}*/

