#include "innoinfo.h"
#include "record.h"
#include "page.h"
#include "util.h"

#include <set>

char *rec_get_infimum(char *page) {
  if(page_index_is_comp(page))
    return page + PAGE_NEW_INFIMUM;
  else
    return page + PAGE_OLD_INFIMUM;
}

char *rec_get_supremum(char *page) {
  if(page_index_is_comp(page))
    return page + PAGE_NEW_SUPREMUM;
  else
    return page + PAGE_OLD_SUPREMUM;
}

char *rec_get_next(int comp, char *page, char *rec) {
  int offset = read_ulshort(rec-2);
  if(offset == 0)
    return NULL;
    
  if(comp) // in compressed page offset is relative
    offset += rec-page;
    
  // may be negative offset
  offset = offset & (PAGE_SIZE-1);
    
  if(offset == PAGE_NEW_SUPREMUM || offset == PAGE_OLD_SUPREMUM || offset < 0 || offset > PAGE_SIZE)
    return NULL;

  // next record points to same record?
  if(page+offset == rec)
    return NULL;

  return page + offset;
}

int field_type_is_fixed(unsigned int type, unsigned int prtype) {
  switch(type) {
    case DATA_MYSQL:
      if(prtype & DATA_BINARY_TYPE) 
        return 0;
      else
        return 1;
    case DATA_VARCHAR:
      return 0;
    case DATA_BINARY:
      return 0;
    case DATA_DECIMAL:
      return 0;
    case DATA_VARMYSQL:
      return 0;
    case DATA_BLOB:
      return 0;
  }
  return 1;
}


char *dump_record(INNORECORD *record) {
  int fields;
  char *field;
  int isnull;
  
  std::vector<INNOFIELD>::iterator i;
  
  printf("Record:\n");
  printf(" Next rec: %d\n", record->next);
  printf(" Offset size: %d bytes\n", record->offsetsize);
  printf(" Fields: %d\n", record->fields);
  printf(" Order nr: %d\n", record->ordernr);
  printf(" Nr rec. owned: %d\n", record->owned);
  printf(" Del. marked: %d\n", record->delmarked);
  
  printf(" Fields: \n");

  for(i=record->field.begin();i!=record->field.end();i++) {
    printf("  NULL: %d\n", i->isnull);

    if(!i->isnull) {
      printf("  Len: %d\n", i->len);
      printf("  blob: %d\n", i->isblob);
      printf("  Data: %s\n", bin2hex(i->data, i->len));	
      if(i->isblob) {
        printf("  blobspace: %d\n", i->blobspace);
        printf("  blobpage: %d\n", i->blobpage);
        printf("  bloboffset: %d\n", i->bloboffset);
        printf("  bloblen: %d\n", i->bloblen);
      }
    }
    printf("\n");
  }
  printf("\n");
}


int rec_validate(INNORECORD *record, char *page) {
  std::vector<INNOFIELD>::iterator iterField;

  for(iterField = record->field.begin(); iterField != record->field.end(); iterField++) {
    if(iterField->data < page || iterField->data >= page + PAGE_SIZE)
      return 0;
    if(iterField->data+iterField->len < page || iterField->data+iterField->len >= page + PAGE_SIZE)
      return 0;
  }
  return 1;
}

INNORECORD *rec_read_old(char *rec, char *page) {
  int i;
  char *fieldptr;
  int isnull;
  int isblob;
  INNORECORD *retval = new INNORECORD;
  
  retval->next = read_ulshort(rec-2);
  retval->offsetsize = read_ulchar(rec-3) & 0x1 ? 1 : 2;
  retval->fields = (( read_ulchar(rec-4) & 0x7 ) << 3) | (read_ulchar(rec-3) >> 1);
  retval->ordernr = (read_ulchar(rec-5) << 8) | (read_ulchar(rec-4) >> 3);
  retval->owned = (read_ulchar(rec-6)  & 0xf);
  retval->delmarked = (read_ulchar(rec-6) >> 4);
  
  fieldptr = rec;
  for(i=0;i<retval->fields;i++) {
    INNOFIELD field;
    int fieldend;
    if(read_ulchar(rec-3) & 0x1) {
      // Single byte offsets
      fieldend = read_ulchar(rec-7-(i));
      isnull = fieldend & 0x80 ? 1 : 0;
      isblob = false;
      fieldend = fieldend & 0x7f;
    } else {
      // Double byte offsets
      fieldend = read_ulshort(rec-8-(i*2));
      isnull = fieldend & 0x8000 ? 1 : 0; 
      isblob = fieldend & 0x4000 ? 1 : 0;
      fieldend = fieldend & 0x3fff;
    }

    field.isnull = isnull;
    field.isblob = isblob;
    if(!isnull) {
      field.len = (rec+fieldend)-fieldptr;
      field.data = fieldptr;
      
      if(isblob) {
        field.blobspace = read_ulint(field.data + field.len - BTR_EXTERN_FIELD_REF_SIZE + BTR_EXTERN_SPACE_ID);
        field.blobpage = read_ulint(field.data + field.len - BTR_EXTERN_FIELD_REF_SIZE + BTR_EXTERN_PAGE_NO);
        field.bloboffset = read_ulint(field.data + field.len - BTR_EXTERN_FIELD_REF_SIZE + BTR_EXTERN_OFFSET);
        field.bloblen = read_dulint(field.data + field.len - BTR_EXTERN_FIELD_REF_SIZE + BTR_EXTERN_LEN);
        
        field.len -= BTR_EXTERN_FIELD_REF_SIZE;
      }
    } else {
      field.len = 0;
    }
      
    fieldptr = rec+fieldend;
    
    retval->field.push_back(field);
  }

  if(!rec_validate(retval, page))
    return NULL;
  
  if(opt_verbose) {
    dump_record(retval);
  }
  
  return retval;
}

INNORECORD *rec_read_new(std::vector<INNOCOLUMN> columns, char *rec, char *page) {
  int fields;
  std::vector<INNOCOLUMN>::iterator i;
  char *fielddata;
  char *offset;
  INNORECORD *record = new INNORECORD;
  
  memset(record, 0, sizeof(INNORECORD));
  
  record->next = read_ulshort(rec-2);
  record->type = read_ulchar(rec-3) & 0x7;
  record->ordernr = read_ulchar(rec-3) & 0x7;
  record->owned = read_ulchar(rec-5) & 0xf;
  record->delmarked = read_ulchar(rec-5) >> 4;
  
  int nullable = dict_get_nullable(columns);
  int nullbytes = (nullable+7)/8;
  int nullablefieldnr = 0;
  char *nullptr = rec-6;
  
  offset = rec-6-nullbytes;
  fielddata = rec;

  for(i=columns.begin();i!=columns.end();i++) {
    INNOFIELD field;
    int fieldend;

    field.isblob = field.isnull = false;
    
    if(!(i->prtype & DATA_NOT_NULL)) {
      // field  may be null, check it
      if(read_ulchar(nullptr) & (1<<nullablefieldnr)) {
        field.isnull = true;
      } else {
        field.isnull = false;
      }
      nullablefieldnr++;
      if(nullablefieldnr > 7) {
        nullablefieldnr = 0;
        nullptr++;
      }
    } else {
      field.isnull = false;
    }
    
    if(!field.isnull) {
      // Not null
      if(field_type_is_fixed(i->mtype, i->prtype)) {
        // Fixed length
        field.len = i->len;
      } else {
        // Variable length, get length
        if(i->len <= 255 && i->mtype != DATA_BLOB) {
          field.len = read_ulchar(offset);
          offset--;
        } else {
          field.len = read_ulchar(offset);
          offset--;
          if(field.len & 0x80) {
            field.isblob = field.len & 0x40 ? 1 : 0;
            field.len = (field.len & ~0xC0) << 8;
            field.len |= read_ulchar(offset);
            offset--;
            
          }
        }
      }
    }  
    
    if(!field.isnull) {
      field.data = fielddata;
      fielddata = fielddata+field.len;

      if(field.isblob) {
        field.blobspace = read_ulint(field.data + field.len - BTR_EXTERN_FIELD_REF_SIZE + BTR_EXTERN_SPACE_ID);
        field.blobpage = read_ulint(field.data + field.len - BTR_EXTERN_FIELD_REF_SIZE + BTR_EXTERN_PAGE_NO);
        field.bloboffset = read_ulint(field.data + field.len - BTR_EXTERN_FIELD_REF_SIZE + BTR_EXTERN_OFFSET);
        field.bloblen = read_dulint(field.data + field.len - BTR_EXTERN_FIELD_REF_SIZE + BTR_EXTERN_LEN);
        
        field.len -= BTR_EXTERN_FIELD_REF_SIZE;
      }

    } else {
      field.data = NULL;
    }
      
    
    record->field.push_back(field);
  }
  record->fields = record->field.size();
  
  if(opt_verbose)
    dump_record(record);
  
  return record;
}



// These functions retrieve the data from a record. The
// field number is the physical on-disk field number, not the user
// field number (field ordering on-disk differs from user field order)
int rec_is_nth_field_null(std::vector<INNOCOLUMN> columns, int comp, char *rec, int field, char *page) {
  INNORECORD *record;
  if(!comp)
    record = rec_read_old(rec, page);
  else
    record = rec_read_new(columns, rec, page);
    
  if(field >= record->fields)
    return 0;
  int retval = record->field[field].isnull;
  delete record;
  return retval;
}

int rec_is_nth_field_blob(std::vector<INNOCOLUMN> columns, int comp, char *rec, int field, char *page) {
  INNORECORD *record;
  if(!comp)
    record = rec_read_old(rec, page);
  else
    record = rec_read_new(columns, rec, page);
    
  if(field >= record->fields)
    return 0;
  int retval = record->field[field].isblob;
  delete record;
  return retval;
}

int rec_get_nth_field_len(std::vector<INNOCOLUMN> columns, int comp, char *rec, int field, char *page) {
  INNORECORD *record;
  if(!comp)
    record = rec_read_old(rec, page);
  else
    record = rec_read_new(columns, rec, page);
    
  if(field >= record->fields)
    return 0;
  int retval = record->field[field].len;
  delete record;
  return retval;
}

char * rec_get_nth_field(std::vector<INNOCOLUMN> columns, int comp, char *rec, int field, char *page) {
  INNORECORD *record;
  if(!comp)
    record = rec_read_old(rec, page);
  else
    record = rec_read_new(columns, rec, page);
    
  if(field >= record->fields)
    return NULL;
  char * retval = record->field[field].data;
  delete record;
  return retval;
}

int rec_get_field_count(std::vector<INNOCOLUMN> columns, int comp, char *rec, char *page) {
  INNORECORD *record;
  if(!comp)
    record = rec_read_old(rec, page);
  else
    record = rec_read_new(columns, rec, page);
    
  int fields = record->fields;
  delete record;
  return fields;
}

int blob_get_data_len(char *pagedata, int offset) {
  if(offset > PAGE_SIZE)
    return 0;
  return read_ulint(pagedata + offset);
}

int blob_get_next_page(char *pagedata, int offset) {
  if(offset > PAGE_SIZE)
    return 0;
  return read_ulint(pagedata + offset+BTR_BLOB_HDR_NEXT_PAGE_NO);
}

// This supports INTs of 1 to 8 bytes. Signed ints are casted to unsigned and returned as unsigned. This means
// that all positive numbers are correct, and negative numbers are correct if you correctly cast back to signed int.
unsigned long long int_from_field(char *data, int len, int uns) {
  unsigned long long tmp;
  int neg = 0;
  char *tmpptr = ((char *)&tmp);
  int n = len;

  memset(&tmp, 0x00, sizeof(tmp));
  
  if(len > sizeof(tmp)) {
    fprintf(stderr,"INT with %d bytes ?? (>%d)\n", len, sizeof(tmp));
    return 0;
  }
  
  data+=len-1;
  
  // Do big-endian -> little-endian
  while(n) {
    *tmpptr = *data;
    n--;
    tmpptr++;
    data--;
  }

  if(!uns) {
    // Signed numbers are stored shifted by half of MAX_INT for that byte size. We just hack that in here
    long long int offset = 0x80;
    offset <<= (len-1)*8;
    tmp = tmp-offset;
  }
  
  return tmp;
}

unsigned long long rec_get_nth_field_int(std::vector<INNOCOLUMN> columns, int comp, char *rec, int field, int uns, char *page) {
  int len = rec_get_nth_field_len(columns, comp, rec, field, page);
  char *data = rec_get_nth_field(columns, comp, rec, field, page);
  
  return int_from_field(data, len, uns);
}

std::string rec_get_nth_field_string(std::vector<INNOCOLUMN> columns, int comp, char *rec, int field, char *page) {
  int len = rec_get_nth_field_len(columns, comp, rec, field, page);
  char *data = rec_get_nth_field(columns, comp, rec, field, page);
  
  std::string d(data, len);
  
  return d;
}

char *dump_record_short(int fd, INNORECORD *record, INNOTABLE *table) {
  int n = 0;
  bool ignorefield;
  
  std::string shorttablename;

  int p = table->name.find("/");
  if(p == -1)
    shorttablename = table->name;
  else
    shorttablename = table->name.substr(p+1);
  
  printf("INSERT INTO %s (", shorttablename.c_str());

  for(std::vector<INNOCOLUMN>::iterator i = table->indexes[0]->columns.begin(); i != table->indexes[0]->columns.end(); i++) {
    if(i->mtype != DATA_SYS && i->mtype != DATA_SYS_CHILD) {
      printf("%s", i->name.c_str());
      if(n < table->indexes[0]->columns.size() - 1)
        printf(",");
    }
    n++;
  }
  
  printf(") VALUES (");
  n = 0;
  
  for(std::vector<INNOFIELD>::iterator i = record->field.begin(); i != record->field.end(); i++) {
    ignorefield = false;
    if(i->isnull)
      printf("NULL");
    else {
      switch(table->indexes[0]->columns[n].mtype) {
        case DATA_VARMYSQL:
        case DATA_MYSQL:
        case DATA_BLOB:
        case DATA_FIXBINARY:
        case DATA_BINARY:
        case DATA_VARCHAR:
        case DATA_CHAR:
          if(!i->isblob)
            printf("'%s'", EscapeBinary(i->data, i->len).c_str());
          else {
            // We do some basic anti-loop protection via setPages
            std::set<unsigned int> setPages;
            int blobpage = i->blobpage;
            int offset = i->bloboffset;
            
            // First, dump data from the index page
            printf("'%s", EscapeBinary(i->data, i->len).c_str());
            
            // Now, dump blob pages until end or until we find a duplicate page
            while(!opt_ignore_blobs && blobpage != -1 && setPages.find(blobpage) == setPages.end()) {
              char *blobpagedata = page_read(fd, blobpage);
              if(blobpagedata == NULL)
                break;
              setPages.insert(blobpage);
              
              // Escape each blob page seperately
              int len = blob_get_data_len(blobpagedata, offset);
              if(offset+8+len > PAGE_SIZE)
                break; // Blob outside page
              printf("%s", EscapeBinary(blobpagedata + offset + 8, len).c_str());
              
              blobpage = blob_get_next_page(blobpagedata, offset);
              offset = FIL_PAGE_DATA;
              page_free(blobpagedata);
            }
            // All done for this blob
            printf("'");
          }
            
          break;
        case DATA_INT: {
          unsigned long long data = int_from_field(i->data, i->len, table->indexes[0]->columns[n].prtype & DATA_UNSIGNED);
          if(table->indexes[0]->columns[n].prtype & DATA_UNSIGNED) {
            printf("%llu", data);
          } else
            printf("%lld", data);
          break;
        }
        case DATA_SYS_CHILD:
        case DATA_SYS:
          ignorefield = true;
          break;
        case DATA_FLOAT:
          if(i->len != sizeof(float))
            printf("BAD FLOAT");
          else
            printf("%f", *(float *)i->data);
          break;
        case DATA_DOUBLE:
          if(i->len != sizeof(double))
            printf("BAD DOUBLE");
          else
            printf("%d", *(double *)i->data);
          break;
        case DATA_DECIMAL:
          printf("%s", i->data);
          break;
      }
    }
    
    if(n < table->indexes[0]->columns.size()-1 && !ignorefield)
      printf(",");
      
    n++;

  }
  printf(");\n");
}

std::string EscapeBinary(char *data, unsigned int len) {
  std::string s;
  int escape;
  while(len) {
    escape = 0;
    switch(*data) {
    case '\0':
      escape = '0';
      break;
    case '\n':
      escape = 'n';
      break;
    case '\r':
      escape = 'r';
      break;
    case '\\':
      escape = '\\';
      break;
    case '\'':
      escape = '\'';
      break;
    case '"':
      escape = '"';
      break;
    case '\032': // apparently this is escaped for some win32 reason ...
      escape = 'Z';
      break;
    }
    
    if(escape) {
      s.append(1, '\\');
      s.append(1, (char) escape);
    } else {
      s.append(1, *data);
    }
    
    data++;
    len--;
  }
  
  return s;
}
