#include "innoinfo.h"
#include "record.h"
#include "page.h"
#include "dict.h"
#include "util.h"

#include <stdio.h>
#include <stdlib.h>

#include <set>

std::list<INNOTABLE*> tables;

const char *col_types[] = { "UNKNOWN_TYPE", "VARCHAR", "CHAR", "FIXBINARY", "BINARY", "BLOB", "INT", "SYS_CHILD", "SYS", "FLOAT", "DOUBLE", "DECIMAL", "VARMYSQL", "MYSQL" };

int dict_get_nullable(std::vector<INNOCOLUMN> columns) {
  std::vector<INNOCOLUMN>::iterator i;
  int n = 0;
  
  for(i=columns.begin(); i!=columns.end(); i++) {
    if(! (i->prtype & DATA_NOT_NULL)) {
      n++;
    }
  }
  
  return n;
}

int table_get_nullable(std::vector<INNOCOLUMN> columns) {
  std::vector<INNOCOLUMN>::iterator i;
  
  int n = 0;
  for(i = columns.begin(); i != columns.end(); i++) {
    if(!(i->prtype & DATA_NOT_NULL))
      n++;
  }
  return n;
}

void load_table_defs(int fd, int indexpage)
{
  if(opt_verbose) {
    printf("Loading table definitions from page %d\n", indexpage);
  }

  char *pagedata = page_read(fd, indexpage);
  char *infimum = rec_get_infimum(pagedata);
  char *record = rec_get_next(0, pagedata, infimum);
  std::vector<INNOCOLUMN> columns; // dummy
  
  while(record) {
    INNOTABLE *table = new INNOTABLE;
    
    table->name = rec_get_nth_field(columns, 0, record, 0, pagedata);
    table->id = rec_get_nth_field_string(columns, 0, record, 3, pagedata);
    table->cols = rec_get_nth_field_int(columns, 0, record, 4, 1, pagedata);
    table->type = rec_get_nth_field_int(columns, 0, record, 5, 1, pagedata);
    table->mixid = rec_get_nth_field_string(columns, 0, record, 6, pagedata);
    table->mixlen = rec_get_nth_field_int(columns, 0, record, 7, 1, pagedata);
    table->clustername = rec_get_nth_field_string(columns, 0, record, 8, pagedata);
    table->space = rec_get_nth_field_int(columns, 0, record, 9, 1, pagedata);
    
    record = rec_get_next(0, pagedata, record);
    
    tables.push_back(table);
  }
  
  if(opt_verbose)
    printf("Loaded %d tables\n", tables.size());
}

void dump_table_defs()
{
  std::list<INNOTABLE*>::iterator i;
  
  for(i=tables.begin();i!=tables.end();i++) {
    INNOTABLE *table = *i;
    printf("Table %s\n", table->name.c_str());
    printf("id: %s\n", bin2hex((char *)table->id.c_str(), table->id.size()));
    printf("cols: %d\n", table->cols);
    printf("type: %d\n", table->type);
    printf("mixid: %s\n", bin2hex((char *)table->mixid.c_str(), table->mixid.size()));
    printf("mixlen: %d\n", table->mixlen);
    printf("clustername: %s\n", bin2hex((char *)table->clustername.c_str(), table->clustername.size()));
    printf("space: %d\n", table->space);
    printf("\n");
    
    printf("Columns:\n");
    std::vector<INNOCOLUMN>::iterator iCol;
    for(iCol = table->columns.begin(); iCol != table->columns.end(); iCol++) {
      INNOCOLUMN column = *iCol;
      printf(" name: %s\n", bin2hex((char*)column.name.c_str(), column.name.size()));
      printf(" mtype: %d\n", column.mtype);
      printf(" prtype: %d\n", column.prtype);
      printf(" len: %d\n", column.len);
      printf(" prec: %d\n", column.prec);
      printf("\n");
    }

    printf("Indexes:\n");
    std::vector<INNOINDEX *>::iterator iInd;
    for(iInd = table->indexes.begin(); iInd != table->indexes.end(); iInd++) {
      INNOINDEX index = **iInd;
      printf(" id: %lld\n", index.id);
      printf(" name: %s\n", bin2hex((char*)index.name.c_str(), index.name.size()));
      printf(" nfields: %d\n", index.nfields);
      printf(" type: %d\n", index.type);
      printf(" space: %d\n", index.space);
      printf(" page: %d\n", index.pageno);
      printf(" columns: (%d) ", index.fieldnames.size());
      
      std::vector<std::string>::iterator iFieldNames;
      for(iFieldNames = index.fieldnames.begin(); iFieldNames != index.fieldnames.end(); iFieldNames++) {
        printf("%s ", iFieldNames->c_str());
      }
      printf("\n");
      
      printf("  Leaf column order: ");
      std::vector<INNOCOLUMN>::iterator iColumn;
      for(iColumn = index.columns.begin(); iColumn != index.columns.end(); iColumn++) {
        printf("%s ", iColumn->name.c_str());
      }
      printf("\n");
      
      printf("  Node column order: ");
      for(iColumn = index.nodecolumns.begin(); iColumn != index.nodecolumns.end(); iColumn++) {
        printf("%s ", iColumn->name.c_str());
      }
      printf("\n");
      
    }
    printf("----------------------------------------\n");
  }
}

void dump_table_defs_short()
{
  std::list<INNOTABLE*>::iterator i;
  
  for(i=tables.begin();i!=tables.end();i++) {
    INNOTABLE *table = *i;
    printf("name: %s ", table->name.c_str());
    printf("type: %d", table->type);
    
    std::vector<INNOCOLUMN>::iterator iCol;
    for(iCol = table->columns.begin(); iCol != table->columns.end(); iCol++) {
      INNOCOLUMN column = *iCol;
      printf(" ,%s", column.name.c_str());
      printf(" %s", col_types[column.mtype]);
      printf(" (%d) ", column.len);
    }
    printf("\n");
  }
}


INNOTABLE *dict_find_table_by_id(std::string tableid) {
  std::list<INNOTABLE*>::iterator i;
  
  for(i=tables.begin(); i != tables.end(); i++) {
    if((*i)->id == tableid)
      return *i;
  }
  return NULL;
}

INNOTABLE *dict_find_table_by_name(std::string tablename) {
  std::list<INNOTABLE*>::iterator i;
  
  for(i=tables.begin(); i != tables.end(); i++) {
    if((*i)->name == tablename)
      return *i;
  }
  return NULL;
}

INNOINDEX *dict_find_index_by_id(unsigned long long int indexid) {
  std::list<INNOTABLE*>::iterator iTable;
  std::vector<INNOINDEX*>::iterator iIndex;
  
  for(iTable = tables.begin(); iTable != tables.end(); iTable++) {
    for(iIndex = (*iTable)->indexes.begin(); iIndex != (*iTable)->indexes.end(); iIndex++) {
      if((*iIndex)->id == indexid) {
        return *iIndex;
      }
    }
  }
  return NULL;
}

void dict_add_column(std::string tableid, unsigned int pos, INNOCOLUMN column)
{
  INNOTABLE *table = dict_find_table_by_id(tableid);
  
  if(table == NULL) {
    printf("Column found for non-existent table %s\n", bin2hex((char*)tableid.c_str(), tableid.size()));
    return;
  }
  
  // Pos is not the index, but just an order. We assume we get the correct order from the caller
  pos = table->columns.size();

  if(table->columns.size() <= pos)
    table->columns.resize(pos+1);
  table->columns[pos]=column;
}

void dict_add_index(std::string tableid, INNOINDEX index) 
{
  INNOTABLE *table = dict_find_table_by_id(tableid);
  
  if(table == NULL) {
    printf("Index found for non-existent table %s\n", bin2hex((char*)tableid.c_str(), tableid.size()));
    return;
  }

  table->indexes.push_back(new INNOINDEX(index));
}

void dict_add_index_field(unsigned long long int indexid, unsigned int pos, std::string colname)
{
  INNOINDEX *index = dict_find_index_by_id(indexid);
  
  // We ignore pos as we assume we get them in ascending order
  pos = index->fieldnames.size();
  
  if(index->fieldnames.size() <= pos)
    index->fieldnames.resize(pos+1);
  index->fieldnames[pos] = colname;
}

void load_column_defs(int fd, int indexpage)
{
  if(opt_verbose) {
    printf("Loading column definitions from page %d\n", indexpage);
  }

  char *pagedata = page_read(fd, indexpage);
  char *infimum = rec_get_infimum(pagedata);
  char *record = rec_get_next(0, pagedata, infimum);
  int n = 0;
  std::vector<INNOCOLUMN> columns; // dummy
  
  while(record) {
    INNOCOLUMN column;
    
    std::string tableid = rec_get_nth_field_string(columns, 0, record, 0, pagedata);
    unsigned int pos = rec_get_nth_field_int(columns, 0, record, 1, 1, pagedata);
    column.name = rec_get_nth_field_string(columns, 0, record, 4, pagedata);
    column.mtype = rec_get_nth_field_int(columns, 0, record, 5, 1, pagedata);
    column.prtype = rec_get_nth_field_int(columns, 0, record, 6, 1, pagedata);
    column.len = rec_get_nth_field_int(columns, 0, record, 7, 1, pagedata);
    column.prec = rec_get_nth_field_int(columns, 0, record, 8, 1, pagedata);
    
    dict_add_column(tableid, pos, column);
    
    record = rec_get_next(0, pagedata, record);
    n++;
  }
  
  if(opt_verbose)
    printf("Loaded %d columns\n", n);
}

void load_index_defs(int fd, int indexpage)
{
  if(opt_verbose) {
    printf("Loading index definitions from page %d\n", indexpage);
  }

  char *pagedata = page_read(fd, indexpage);
  char *infimum = rec_get_infimum(pagedata);
  char *record = rec_get_next(0, pagedata, infimum);
  int n = 0;
  std::vector<INNOCOLUMN> columns; // dummy

  while(record) {
    INNOINDEX index;
    
    std::string tableid = rec_get_nth_field_string(columns, 0, record, 0, pagedata);
    index.id = rec_get_nth_field_int(columns, 0, record, 1, 1, pagedata);
    index.name = rec_get_nth_field_string(columns, 0, record, 4, pagedata);
    index.nfields = rec_get_nth_field_int(columns, 0, record, 5, 1, pagedata);
    index.type = rec_get_nth_field_int(columns, 0, record, 6, 1, pagedata);
    index.space = rec_get_nth_field_int(columns, 0, record, 7, 1, pagedata);
    index.pageno = rec_get_nth_field_int(columns, 0, record, 8, 1, pagedata);
    
    dict_add_index(tableid, index);
    
    record = rec_get_next(0, pagedata, record);
    n++;
  }  
  
  if(opt_verbose)
    printf("Loaded %d indexes\n", n);
}

void load_index_field_defs(int fd, int indexpage)
{
  if(opt_verbose) {
    printf("Loading index field definitions from page %d\n", indexpage);
  }

  char *pagedata = page_read(fd, indexpage);
  char *infimum = rec_get_infimum(pagedata);
  char *record = rec_get_next(0, pagedata, infimum);
  int n = 0;
  std::vector<INNOCOLUMN> columns; // dummy
  
  while(record) {
    unsigned long long int indexid = rec_get_nth_field_int(columns, 0, record, 0, 1, pagedata);
    int pos = rec_get_nth_field_int(columns, 0, record, 1, 1, pagedata);
    std::string colname = rec_get_nth_field_string(columns, 0, record, 4, pagedata);
    
    dict_add_index_field(indexid, pos, colname);
    
    record = rec_get_next(0, pagedata, record);
    n++;
  }
  
  if(opt_verbose)
    printf("Loaded %d index fields\n", n);
}

INNOCOLUMN find_field_by_name(INNOTABLE *table, std::string colname)
{
  INNOCOLUMN defcol;
  
  std::vector<INNOCOLUMN>::iterator iColumns;
  
  for(iColumns = table->columns.begin(); iColumns != table->columns.end(); iColumns++) {
    if(iColumns->name == colname) {
      return *iColumns;
    }
  }
  
  fprintf(stderr, "WARNING: unable to find column %s which is in the primary key!\n", colname.c_str());
  
  // This shouldn't happen. If it does, we're assuming the missing column was an int. You'll
  // be very lucky to read any data from your database if you have this problem ...
  defcol.name = "UNKNOWN";
  defcol.mtype = DATA_INT;
  defcol.prtype = DATA_UNSIGNED;
  defcol.len = 4;
  
  return defcol;
}

// This function doesn't really load anything, but generates the column order for the indexes.
void load_index_columns()
{
  if(opt_verbose) {
    printf("Generating on-disk column order..\n");
  }
  
  std::list<INNOTABLE*>::iterator iTables;
  std::vector<INNOINDEX *>::iterator iIndexes;
  std::vector<std::string>::iterator iFieldnames;
  std::vector<INNOCOLUMN>::iterator iColumns;
  INNOTABLE *table = NULL;
  INNOINDEX *index = NULL;
  
  INNOCOLUMN rowidcol, trxcol, undocol, pagenocol;
  
  rowidcol.name = "ROW_ID";
  rowidcol.mtype = DATA_SYS;
  rowidcol.prtype = DATA_ROW_ID | DATA_NOT_NULL;
  rowidcol.len = DATA_ROW_ID_LEN;
  
  trxcol.name = "TRX";
  trxcol.mtype = DATA_SYS;
  trxcol.prtype = DATA_TRX_ID | DATA_NOT_NULL;
  trxcol.len = DATA_TRX_ID_LEN;
  
  undocol.name = "UNDO";
  undocol.mtype = DATA_SYS;
  undocol.prtype = DATA_ROLL_PTR | DATA_NOT_NULL;
  undocol.len = DATA_ROLL_PTR_LEN;
  
  pagenocol.name = "PAGE_NO";
  pagenocol.mtype = DATA_SYS_CHILD;
  pagenocol.prtype = DATA_NOT_NULL;
  pagenocol.len = 4;
  
  for(iTables = tables.begin(); iTables != tables.end(); iTables++) {
    table = *iTables;
    for(iIndexes = table->indexes.begin(); iIndexes != table->indexes.end(); iIndexes++) {
      std::set<std::string> setInIndex;
      index = *iIndexes;
      
      // First, indexed columns
      for(iFieldnames = index->fieldnames.begin(); iFieldnames != index->fieldnames.end(); iFieldnames++) {
        INNOCOLUMN col = find_field_by_name(table, *iFieldnames);
        
        index->columns.push_back(col);
        index->nodecolumns.push_back(col);
        
        // Remember this column was in the index
        setInIndex.insert(*iFieldnames);
      }
      
      if(setInIndex.size() == 0) {
        // If there are no primary keys, then innodb generates a 6-byte row id
        index->columns.push_back(rowidcol);
      }
      
      if(index->type & DICT_CLUSTERED) {
        // Then, two standard columns
        index->columns.push_back(trxcol);
        index->columns.push_back(undocol);

        // The clustered index contains all other columns after the primary key columns
        for(iColumns = table->columns.begin(); iColumns != table->columns.end(); iColumns++) {
          if(setInIndex.find(iColumns->name) != setInIndex.end())
            continue;
          
          index->columns.push_back(*iColumns);
        }
      } else {
        // A secondary index contains the primary's columns after its own primary
        for(iFieldnames = table->indexes[0]->fieldnames.begin(); iFieldnames != table->indexes[0]->fieldnames.end(); iFieldnames++) {
          INNOCOLUMN col = find_field_by_name(table, *iFieldnames);
        
          index->columns.push_back(col);
        }
      }
      
      // In node pages, the primary's are followed by a single int pageno
      index->nodecolumns.push_back(pagenocol);
    }
  }
}

