#!/bin/sh

#
# Zarafa RHEL LDAP user manager install script
# (c) 2004-2007 Zarafa, The Netherlands
#

VERSION=SVNVER
RELEASE=SVNREV
INSTALL=1
if [ ! -z "$1" -a "$1" == "-config" ]; then
	INSTALL=0
fi

distro_text() {
	echo
	echo "If you require installation on another system, this is only"
	echo "possible via manual installation. Please follow installation"
	echo "instructions in the installation documentation."
	echo
}

warning() {
	echo
	echo "*** WARNING ***"
	echo
}

continue_anyway() {
	echo "If you still wish to continue with this installation script, please make"
	echo "sure the Zarafa packages are already installed on the system."
	echo "Press <enter> to continue, or <ctrl-c> to stop."
	read dummy
}

continue_install() {
    echo ""
	echo "This script will configure your OpenLDAP installation"
	echo "Press <enter> to continue, or <ctrl-c> to stop."
	read dummy
}

continue_postfix_install() {
    echo ""
	echo "This script will configure your Postfix installation for local mail delivery"
	echo "Press <enter> to continue, or <ctrl-c> to stop."
	read dummy
}

redhat45_only() {
	warning
	echo "This installation script only runs with RedHat Enterprise Linux 4 and 5."
	echo "Your distribution was found to be:"
	cat /etc/redhat-release
	distro_text
	continue_anyway
}

unsupported_distro() {
	warning
	echo "Unfortunately, this installation script could not determine"
	echo "your Linux Distribution. Please follow installation"
	echo "instructions in the installation documentation for a manual installation."
	echo
	continue_anyway
	ARCH="Unknown"
	DISTRO="Unknown"
	DVERSION="Unknown"
}

unsupported_architecture() {
	echo "Unfortunately, you use an unsupported architecture for this distribution."
	echo
	exit 1
}

openldap_missing() {
	echo "Zarafa for RHEL requires OpenLDAP 2.2.13 or higher."
	echo "Please install OpenLDAP and restart the installer."
	echo
	exit 1
}

openldap_servers_missing() {
	echo "Zarafa for RHEL requires openldap-servers package."
	echo "Please install openldap-servers from the Servers - Network Servers Group"
	echo "and restart the installer."
	echo
	exit 1
}

openldap_clients_missing() {
	echo "Zarafa for RHEL requires openldap-clients package."
	echo "Please install openldap-clients from the System - System Tools Group"
	echo "and restart the installer."
	echo
	exit 1
}

python_ldap_missing() {
	echo "Zarafa for RHEL requires python-ldap package."
	echo "Please install python-ldap from the Development - Development Tools Group"
	echo "and restart the installer."
	echo
	exit 1
}

postfix_missing() {
	echo "Zarafa for RHEL requires the postfix MTA package."
	echo "Please install postfix from the Servers - Mail Server Package Group"
	echo "and restart the installer."
	echo
	exit 1
}

openldap_already_configured() {
	echo "Your OpenLDAP installation appears to already be configured"
	echo "If you wish to reinstall the LDAP base installation you will need to"
	echo "stop OpenLDAP, delete the files in /var/lib/ldap/ and restart the installer."
	echo "The installer will exit now."
	echo
	exit 1
}

postfix_already_configured() {
	echo "Your Postfix installation appears to already be configured"
	echo "If you wish to reinstall the Postfix base installation you will need to"
	echo "stop postfix, switch to a different mailer (i.e. using system-switch-mail)"
	echo "and restart the installer."
	echo "Please also read the Ldap postfix whitepaper document for more information."
	echo "The installer will exit now."
	echo
	exit 1
}

distro_check() {
	DISTRO=""
#
# Redhat (RHEL) check
#
	if [ -f /etc/redhat-release ]; then
		DISTRO=RedHat
		DVERSION=`sed -e 's/.*release \([[:digit:]]\).*/\1/' /etc/redhat-release`
		ARCH=`rpm -q rpm --queryformat '%{ARCH}'`

		if [ $DVERSION -ne 4 -a $DVERSION -ne 5 ]; then
			redhat45_only
		fi
		if [ $ARCH != "i386" -a $ARCH != "x86_64" ]; then
			unsupported_architecture
		fi
		return
	fi

	unsupported_distro
}

replace() {
	file=$1
	option=$2
	value=$3

	# not completely flawless
	if echo $value | grep -v '\\%' | grep -q '%'; then
		echo "Please type \\\\ before the % sign for the option $option"
		echo -n "$option: "
		read value
	fi

	sed -i -e "s%\([[:space:]]*$option[[:space:]]*=[[:space:]]*\).*%\1 $value%" $file
}

replace_slapd() {
	file=$1
	option=$2
	value=$3

	# not completely flawless
	if echo $value | grep -v '\\%' | grep -q '%'; then
		echo "Please type \\\\ before the % sign for the option $option"
		echo -n "$option: "
		read value
	fi

	sed -i -e "s%\(^$option[[:space:]]*\).*%\1$value%" $file
}

uncomment() {
	file=$1
	option=$2

	sed -i -e "s%[[:space:]]*#\(.*$option.*\)%\1%" $file
}

uncomment_slapd_rootpw() {
	file=$1

	sed -i -e "s%^#.*\(rootpw.*secret.*\)%\1%" $file
}

append_slapd() {
	file=$1
	soption=$2
	svalue=$3
	aoption=$4
	avalue=$5

	sed -i -e "s%\(^$soption\)\([[:space:]]*\)\($svalue.*\)%\1\2\3\n$aoption\2$avalue%" $file
}

append_option() {
	file=$1
	soption=$2
	svalue=$3
	aoption=$4
	avalue=$5

	sed -i -e "s%\(^$soption\)\([[:space:]]*\)=\([[:space:]]*\)\($svalue.*\)%\1\2=\3\4\n$aoption\2=\3$avalue%" $file
}


echo 
echo "Zarafa Outlook Sharing, version $VERSION-$RELEASE"
echo "(C) Copyright 2004-2006 Zarafa, The Netherlands"
echo "Please visit http://www.zarafa.com/ for more information."
echo


distro_check

    if [ $INSTALL == 1 ]; then
#
# RPM packages
#
		if ! rpm -q zarafa-config-users &> /dev/null; then
			if ! rpm -U -h zarafa-config-users-*.noarch.rpm; then
				echo
				echo "Please resolve above errors from rpm, and restart the installer."
				exit 1
			fi
		else
			echo "Already installed, skipping packages."
		fi
	fi

	echo
	echo "Installation of the software is complete."

#
# Check if openldap is installed and not yet configured
#

	rpm -q openldap >/dev/null
	if [ $? -ne 0 ]; then
		openldap_missing
	fi

	rpm -q openldap-servers >/dev/null
	if [ $? -ne 0 ]; then
		openldap_servers_missing
	fi

	rpm -q openldap-clients >/dev/null
	if [ $? -ne 0 ]; then
		openldap_clients_missing
	fi

	rpm -q python-ldap >/dev/null
	if [ $? -ne 0 ]; then
		python_ldap_missing
	fi
	
	rpm -q postfix >/dev/null
	if [ $? -ne 0 ]; then
		postfix_missing
	fi

	if [ $(ls -1 /var/lib/ldap/ | wc -l) != 0 ]; then
		openldap_already_configured
	fi


# last chance to bail out

	continue_install

	echo "Configuring OpenLDAP ..."
	echo ""
	while true; do
		echo "Please enter your domain"
		echo -ne " Domain []\t\t: "
		read DOMAIN
		break
	done
	
	DN=`echo "$DOMAIN" | sed -e 's/\./,dc=/g' -e 's/\(.*\)/dc=\1/'`
	DC=`echo "$DOMAIN" | sed -e 's/\([^\.]*\).*/\1/'`
	echo "Your LDAP base DN will be: $DN"
	echo ""

	while true; do
		echo "Please enter the password for LDAP Manager"
		echo -ne " Password []\t\t: "
		read LDAPPASS
		echo -ne " Confirm  []\t\t: "
		read LDAPPASSCONFIRM
		if [ "$LDAPPASS" != "$LDAPPASSCONFIRM" ]; then
			echo "Your Passwords do not match each other, please reenter..."
		else
			break
		fi
	done

	uncomment_slapd_rootpw /etc/openldap/slapd.conf

	
	replace_slapd /etc/openldap/slapd.conf rootpw `slappasswd -s "$LDAPPASS"`
	replace_slapd /etc/openldap/slapd.conf suffix '"'$DN'"'
	replace_slapd /etc/openldap/slapd.conf rootdn '"cn=Manager,'$DN'"'

# enable LDAP in server.cfg
	echo "Enabling LDAP in /etc/zarafa/server.cfg ..."
	replace /etc/zarafa/server.cfg user_plugin ldap
	
	echo ""

	cp -p /usr/share/zarafa/example.ldap.cfg /etc/zarafa/ldap.cfg
	replace /etc/zarafa/ldap.cfg ldap_bind_user 'cn=Manager,'$DN
	replace /etc/zarafa/ldap.cfg ldap_bind_passwd "$LDAPPASS"
	replace /etc/zarafa/ldap.cfg ldap_user_search_base 'ou=Users,'$DN
	replace /etc/zarafa/ldap.cfg ldap_user_search_filter "(objectClass=inetorgperson)"
	replace /etc/zarafa/ldap.cfg ldap_user_unique_attribute uid
	replace /etc/zarafa/ldap.cfg ldap_group_search_base 'ou=Groups,'$DN
	replace /etc/zarafa/ldap.cfg ldap_group_search_filter "(objectClass=groupofuniquenames)"
	replace /etc/zarafa/ldap.cfg ldap_group_unique_attribute cn
	replace /etc/zarafa/ldap.cfg ldap_groupmembers_attribute uniqueMember
	replace /etc/zarafa/ldap.cfg ldap_groupmembers_attribute_type dn
	replace /etc/zarafa/ldap.cfg ldap_quota_multiplier 1048576

	cp -p /usr/share/zarafa-config-users/example.ldapadmin.cfg /etc/zarafa/ldapadmin.cfg

	echo "Installing zarafa.schema..."
	cp -p /usr/share/zarafa/zarafa.schema /etc/openldap/schema/
	# check if zarafa.schema is not yet included
	grep zarafa.schema /etc/openldap/slapd.conf >/dev/null
	if [ $? -ne 0 ]; then
		append_slapd /etc/openldap/slapd.conf include /etc/openldap/schema/nis.schema include /etc/openldap/schema/zarafa.schema
	fi

	echo "Installing qmail.schema..."
	cp -p /usr/share/zarafa-config-users/qmail.schema /etc/openldap/schema/
	# check if qmail.schema is not yet included
	grep qmail.schema /etc/openldap/slapd.conf >/dev/null
	if [ $? -ne 0 ]; then
		append_slapd /etc/openldap/slapd.conf include /etc/openldap/schema/zarafa.schema include /etc/openldap/schema/qmail.schema
	fi
	
	replace_slapd /etc/openldap/ldap.conf BASE $DN

	chown ldap.ldap /etc/openldap/slapd.conf
	chown ldap.ldap /etc/openldap/ldap.conf

	/etc/init.d/ldap start

	echo "Waiting for OpenLDAP to start ..."
	sleep 10
	
	echo "Creating Base DNs ..."
	
	echo "dn: $DN
objectclass: dcObject
objectclass: organization
o: $DC
# dc ist der erste Teil der dn
dc: $DC

dn: cn=Manager,$DN
objectclass: organizationalRole
cn: Manager

# Eintrag 3: ou=Groups,$DN
dn:ou=Groups,$DN
ou: Groups
objectClass: top
objectClass: organizationalUnit

# Eintrag 4: ou=Users,$DN
dn:ou=Users,$DN
ou: Users
objectClass: top
objectClass: organizationalUnit

" | ldapadd -x -D "cn=Manager,$DN" -w "$LDAPPASS"

	echo ""
	
	echo "Configuring automatic ldap startup at boot time ..."
	
	chkconfig --add ldap
	chkconfig ldap on
	
	echo "Restarting zarafa ..."
	
	/etc/init.d/zarafa-server restart
	/etc/init.d/zarafa-spooler restart
	/etc/init.d/zarafa-monitor restart
	/etc/init.d/zarafa-gateway restart

    file /etc/alternatives/mta | grep postfix >/dev/null
	if [ $? -eq 0 ]; then
		postfix_already_configured
	fi
	
    continue_postfix_install
    
    echo ""
    echo "Configuring postfix as the MTA ..."
    
    /usr/sbin/alternatives --set mta /usr/sbin/sendmail.postfix  
    /sbin/service sendmail stop 2>/dev/null >/dev/null
    /sbin/service exim stop 2>/dev/null >/dev/null
    
    MAINCF=/etc/postfix/main.cf
    uncomment $MAINCF "mydomain ="
    replace $MAINCF mydomain "$DOMAIN"
    
    #grep "^mailbox_command" $MAINCF >/dev/null
	#if [ $? -ne 0 ]; then
	#    append_option $MAINCF "#mailbox_command" "/some/where/procmail -a" "mailbox_command" "/usr/bin/zarafa-dagent -v \$USER"
	#fi
    
    VMAILUSER=vmail
    VMAILHOME=/var/spool/$VMAILUSER
    /usr/bin/id $VMAILUSER >/dev/null 2>/dev/null
	if [ $? -ne 0 ]; then
        /usr/sbin/useradd -r -d $VMAILHOME -m -s /sbin/nologin $VMAILUSER 
    fi
    VMAILUID=`id -u $VMAILUSER`
    VMAILGID=`id -g $VMAILUSER`
    
    grep "virtual_transport = zarafa" $MAINCF >/dev/null
	if [ $? -ne 0 ]; then
	    echo "virtual_transport = zarafa:
zarafa_destination_recipient_limit = 1
virtual_mailbox_domains = $DOMAIN
virtual_mailbox_base = /home/vmail
virtual_uid_maps = static:$VMAILUID
virtual_gid_maps = static:$VMAILGID
virtual_minimum_uid = $VMAILUID

virtual_mailbox_maps = ldap:ldap-users
ldap-users_server_host = localhost
ldap-users_search_base = ou=Users,$DN
ldap-users_scope = sub
ldap-users_query_filter = (mail=%s)
ldap-users_result_attribute = uid

virtual_alias_maps = ldap:ldap_virtual_alias
ldap_virtual_alias_server_host = localhost
ldap_virtual_alias_search_base = ou=Users,$DN
ldap_virtual_alias_scope = sub
ldap_virtual_alias_query_filter = (mailAlternateAddress=%s)
ldap_virtual_alias_result_attribute = mail
" >>$MAINCF

    fi
	
	MASTERCF=/etc/postfix/master.cf
    grep "zarafa" $MASTERCF >/dev/null
	if [ $? -ne 0 ]; then
	    echo "zarafa    unix  -       n       n       -       5       pipe" >>$MASTERCF
        echo "        flags=DRhu user=vmail argv=/usr/bin/zarafa-dagent \${mailbox}" >>$MASTERCF
    fi
        
    replace /etc/zarafa/server.cfg local_admin_users "root vmail"
        
	echo "Configuring automatic postfix startup at boot time ..."
	
	chkconfig --add postfix
	chkconfig postfix on

    echo "Starting postfix ..."
    /sbin/service postfix start

	echo "Restarting zarafa server..."
	
	/etc/init.d/zarafa-server restart

    echo ""
	echo "DONE. You can now add users with zarafa-config-users."

	
