## mainWindow.py - main interface window for redhat-config-users
## Copyright (C) 2001-2003 Red Hat, Inc.
## Copyright (C) 2001-2003 Brent Fox <bfox@redhat.com>
## Copyright (C) 2006-2007 Zarafa LDAP extensions by Zarafa, The Netherlands

## This program is free software; you can redistribute it and/or modify
## it under the terms of the GNU General Public License as published by
## the Free Software Foundation; either version 2 of the License, or
## (at your option) any later version.

## This program is distributed in the hope that it will be useful,
## but WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
## GNU General Public License for more details.

## You should have received a copy of the GNU General Public License
## along with this program; if not, write to the Free Software
## Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

## Author: Brent Fox

import signal
import gtk
import gobject
import gtk.glade
import string
import sys
import time
import os
import rpm

#import libuser
import libuser_zarafa_ldap
libuser = libuser_zarafa_ldap
import groupWindow
import userWindow
import userProperties
import groupProperties
import messageDialog

##
## I18N
##
from rhpl.translate import _, N_
import rhpl.translate as translate
import rhpl.executil
domain = "zarafa-config-users"
translate.textdomain (domain)
gtk.glade.bindtextdomain(domain)

def service_pending_events():
    while gtk.events_pending():
        if gtk.__dict__.has_key ("main_iteration"):
            gtk.main_iteration()
        else:
            gtk.mainiteration()

def admin_prompt_callback(prompts, title):
    dialog = gtk.Dialog()
    dialog.set_title(title)
    dialog.set_position(gtk.WIN_POS_CENTER)
    dialog.set_icon(iconPixbuf)
    dialog.add_button(gtk.STOCK_OK, 1)
    dialog.add_button(gtk.STOCK_CANCEL, 0)

    table = gtk.Table(rows = len(prompts), columns = 2)
    dialog.vbox.pack_start(table)

    table.set_row_spacings(4)
    table.set_col_spacings(4)

    ret_list = []
    for i in range(len(prompts)):
        prompt = prompts[i]

        label = gtk.Label(prompt.prompt)
        label.set_alignment(1.0, 0.5)
        table.attach(label, 0, 1, i, i + 1)

        entry = gtk.Entry()
        entry.set_visibility(prompt.visible)
        if prompt.default_value:
            entry.set_text(prompt.default_value)
        table.attach(entry, 1, 2, i, i + 1)

        ret_list.append((prompt, entry))
        
    table.show_all()

    i = dialog.run()
    if i == 1:
        for (prompt, entry) in ret_list:
            prompt.value = entry.get_text()
        dialog.destroy()
        return gtk.TRUE

    sys.exit(1)


gtk.glade.bindtextdomain(domain)
if os.access("zarafa-config-users.glade", os.F_OK):
    xml = gtk.glade.XML ("./zarafa-config-users.glade", domain=domain)
else:
    xml = gtk.glade.XML ("/usr/share/zarafa-config-users/zarafa-config-users.glade", domain=domain)

busy_cursor = gtk.gdk.Cursor(gtk.gdk.WATCH)
ready_cursor = gtk.gdk.Cursor(gtk.gdk.LEFT_PTR)

##
## Icon for windows
##

iconPixbuf = None      
try:
    iconPixbuf = gtk.gdk.pixbuf_new_from_file("/usr/share/zarafa-config-users/zarafa-config-users.png")
except:
    pass

class mainWindow:

    def version():
        # substituted to the real version by the Makefile at installation time.
        return "1.2.27"

    def destroy (self, *args):
        settings = '/etc/sysconfig/zarafa-config-users'
        filterLineFound = gtk.FALSE
        
        #try:
        if 0:
            fd = open(settings, 'w')

            if self.configFile:
                for line in self.configFile:
                    if line == '' or line[0] == '#':
                        fd.write(line)
                    else:
                        tokens = string.split(line, '=')
                        if tokens[0] == 'FILTER':
                            filterLineFound = gtk.TRUE
                            if self.filterSystemUsers == gtk.TRUE:
                                fd.write("FILTER=true\n")
                            else:
                                fd.write("FILTER=false\n")

                if filterLineFound == gtk.FALSE:
                    #For some reason, the config file is messed up.  Let's write out a new FILTER line
                    if self.filterSystemUsers == gtk.TRUE:
                        fd.write("FILTER=true\n")
                    else:
                        fd.write("FILTER=false\n")
                    
            else:
                fd.write("#Configuration file for zarafa-config-users\n")
                fd.write("\n")
                fd.write("#Filter out system users by default\n")
                fd.write("FILTER=true\n")
            
            fd.close()
        #except IOError:
        #    print _("Error saving settings to %s") % settings
        if gtk.__dict__.has_key ("main_quit"):
            gtk.main_quit()
        else:
            gtk.mainquit()

    def __init__(self):
        self.readPreferencesFile()
        nameTag = _("Users and Groups")
        commentTag = _("Add or remove users and groups")
        
        self.toplevel = xml.get_widget ('mainWindow')
        self.toplevel.resize(775, 550)
        self.toplevel.set_icon(iconPixbuf)
        self.add_user_button = xml.get_widget("add_user_button")
        self.add_group_button = xml.get_widget("add_group_button")
        self.properties_button = xml.get_widget("properties_button")
        self.properties_menu = xml.get_widget("properties_menu")
        self.delete_button = xml.get_widget("delete_button")
        self.delete_menu = xml.get_widget("delete_menu")
        self.toplevel.connect("destroy", self.destroy)
        
        import messageDialog
        import userGroupCheck
        def logonCallback(user):
            return messageDialog.input_prompt(_("Password:"), None, _("LDAP Logon for %s" % user), userGroupCheck.returnOK)

        self.ADMIN = libuser_zarafa_ldap.admin(logonCallback, _("LDAP Access"), _("Password Required")) 
        #prompt=admin_prompt_callback, prompt_data=(_('Logon Information Required'),))
        #self.ADMIN = libuser.admin(prompt=admin_prompt_callback, prompt_data=(_('Logon Information Required'),))

        self.userStore = gtk.ListStore(gobject.TYPE_STRING, gobject.TYPE_INT, gobject.TYPE_STRING,
                                       gobject.TYPE_STRING, gobject.TYPE_STRING, gobject.TYPE_STRING,
                                       gobject.TYPE_PYOBJECT)
        self.userStore.set_sort_column_id(1, gtk.SORT_ASCENDING)        
        self.userTreeView = xml.get_widget('user_view')
        self.userTreeView.set_rules_hint(gtk.TRUE)
        self.userTreeView.set_model(self.userStore)
        self.userTreeView.get_selection().connect("changed", self.itemSelected)
        self.userTreeView.connect("row_activated", self.rowActivated)

        col = gtk.TreeViewColumn(_("User Name"), gtk.CellRendererText(), text=0)
        col.set_sort_column_id(0)
        col.set_resizable(gtk.TRUE)
        self.userTreeView.append_column(col)
        #col = gtk.TreeViewColumn(_("User ID"), gtk.CellRendererText(), text=1)
        #col.set_sort_column_id(1)
        #col.set_resizable(gtk.TRUE)
        #self.userTreeView.append_column(col)
        #col = gtk.TreeViewColumn(_("Primary Group"), gtk.CellRendererText(), text=2)
        #col.set_sort_column_id(2)
        #col.set_resizable(gtk.TRUE)
        #self.userTreeView.append_column(col)
        col = gtk.TreeViewColumn(_("Full Name"), gtk.CellRendererText(), text=3)
        col.set_sort_column_id(3)
        col.set_resizable(gtk.TRUE)
        self.userTreeView.append_column(col)
#        col = gtk.TreeViewColumn(_("Login Shell"), gtk.CellRendererText(), text=4)
        col = gtk.TreeViewColumn(_("E-Mail"), gtk.CellRendererText(), text=4)
        col.set_sort_column_id(4)
        col.set_resizable(gtk.TRUE)
        self.userTreeView.append_column(col)
        col = gtk.TreeViewColumn(_("Group memberships"), gtk.CellRendererText(), text=5)
        col.set_sort_column_id(5)
        col.set_resizable(gtk.TRUE)
        self.userTreeView.append_column(col)

        self.userTreeView.set_property("headers-visible", gtk.TRUE)

        self.groupStore = gtk.ListStore(gobject.TYPE_STRING, gobject.TYPE_INT, gobject.TYPE_STRING,
                                        gobject.TYPE_PYOBJECT)
        self.groupStore.set_sort_column_id(1, gtk.SORT_ASCENDING)
        self.groupTreeView = xml.get_widget('group_view')
        self.groupTreeView.set_rules_hint(gtk.TRUE)
        self.groupTreeView.set_model(self.groupStore)
        self.groupTreeView.get_selection().connect("changed", self.itemSelected)
        self.groupTreeView.connect("row_activated", self.rowActivated)
        
        col = gtk.TreeViewColumn(_("Group Name"), gtk.CellRendererText(), text=0)
        col.set_sort_column_id(0)
        col.set_resizable(gtk.TRUE)
        self.groupTreeView.append_column(col)
#        col = gtk.TreeViewColumn(_("Group ID"), gtk.CellRendererText(), text=1)
#        col.set_sort_column_id(1)
#        col.set_resizable(gtk.TRUE)
#        self.groupTreeView.append_column(col)
        col = gtk.TreeViewColumn(_("Group Members"), gtk.CellRendererText(), text=2)
        col.set_resizable(gtk.TRUE)
        self.groupTreeView.append_column(col)

        self.notebook = xml.get_widget('notebook1')
        self.notebook.connect('switch-page', self.changeNotebookPage)
        self.filter = xml.get_widget('filterEntry')

        self.user_dict = {}
        self.group_dict = {}
        self.gid_dict = {}
        self.member_dict = {}
        self.service_interval = 10

        selinuxEnabled = self.isSELinuxEnabled()

        self.userWin = userWindow.userWindow(self, self.userStore, self.groupTreeView, xml, selinuxEnabled)
        self.groupWin = groupWindow.groupWindow(self, self.userTreeView, self.groupTreeView, xml)
        self.userProperties = userProperties.userProperties(self, self.userTreeView, self.groupTreeView, xml, selinuxEnabled)
        self.groupProperties = groupProperties.groupProperties(self, self.userTreeView, self.groupTreeView, xml)
        self.userSelectedRow = None
        self.groupSelectedRow = None
        self.toplevel.show_all ()

        xml.signal_connect("on_about_activate", self.on_about_button_clicked)
        xml.signal_connect("on_manual_activate", self.on_manual_button_clicked)
        xml.signal_connect("on_add_user_activate", self.on_add_user_activate)
        xml.signal_connect("on_add_user_button_clicked", self.on_add_user_activate)
        xml.signal_connect("on_add_group_activate", self.on_add_group_activate)
        xml.signal_connect("on_add_group_button_clicked", self.on_add_group_activate)
        xml.signal_connect("on_properties_activate", self.on_properties_activate)
        xml.signal_connect("on_properties_button_clicked", self.on_properties_activate)
        xml.signal_connect("on_delete_activate", self.on_delete_activate)
        xml.signal_connect("on_delete_button_clicked", self.on_delete_activate)
        xml.signal_connect("on_help_button_clicked", self.on_help_button_clicked)
        xml.signal_connect("on_new_user_activate", self.on_add_user_activate)
        xml.signal_connect("on_filterButton_clicked", self.refresh)
        xml.signal_connect("on_filterEntry_activate", self.refresh)
        xml.signal_connect("on_refreshButton_clicked", self.refresh)
        xml.signal_connect("on_exit_activate", self.on_exit_activate)

        self.refresh()
        self.ready()
        if gtk.__dict__.has_key ("main"):
            gtk.main ()
        else:
            gtk.mainloop ()

    def readPreferencesFile(self):
        self.configFile = None
        self.filterSystemUsers = gtk.FALSE
        try:
            fd = open('/etc/sysconfig/zarafa-config-users', 'r')
            self.configFile = fd.readlines()
            fd.close()

            for line in self.configFile:
                line = string.strip(line)
                if line != '' and line[0] != '#':
                    tokens = string.split(line, '=')
                    if tokens[0] == "FILTER":
                        if tokens[1] == "true":
                            self.filterSystemUsers = gtk.TRUE
                        else:
                            self.filterSystemUsers = gtk.FALSE
        except:
            #no config file found.  assume default as gtk.TRUE            
            self.filterSystemUsers = gtk.FALSE

    def on_exit_activate(self, args):
        self.destroy()

    def busy(self):
        self.toplevel.set_sensitive(gtk.FALSE)
        self.toplevel.window.set_cursor(busy_cursor)
        pass

    def ready(self):
        self.toplevel.window.set_cursor(ready_cursor)
        self.toplevel.set_sensitive(gtk.TRUE)
        pass

    def get_user_list (self, filter):
        if filter == "":
            users = self.ADMIN.enumerateUsersFull()
        else:
            users = self.ADMIN.enumerateUsersFull(filter)

        users.sort()
        return users

    def get_group_list(self, filter):
        if filter == "":
            groups = self.ADMIN.enumerateGroupsFull()
        else:
            groups = self.ADMIN.enumerateGroupsFull(filter)
        groups.sort()

        return groups

    def refresh_users(self):
        self.user_dict = {}
        self.gid_dict = {}

        # pull up information about users and store the objects in
	# a dictionary keyed by user name
        i = 0
        for user in self.get_user_list(self.get_filter_data()):
            i = i + 1
            if i >= self.service_interval:
                service_pending_events()
                i = 0

            userName = user.get(libuser.USERNAME)[0]
            self.user_dict[userName] = user

            # try to get a name to associate with the user's primary gid,
            # and attempt to minimize lookups by caching answers
            try:
                gidNumber = user.get(libuser.GIDNUMBER)[0]
            except:
                pass

            try:
                group = self.gid_dict[long (gidNumber)]
            except:
                try:
                    group = self.ADMIN.lookupGroupById(long (gidNumber))
                    self.gid_dict[long (gidNumber)] = group
                except:
                    pass

    def refresh_groups(self):
        self.group_dict = {}
        self.member_dict = {}

        i = 0
        for group in self.get_group_list(self.get_filter_data()):
            i = i + 1
            if i >= self.service_interval:
                service_pending_events()
                i = 0

            groupName = group.get(libuser.GROUPNAME)[0]
            self.group_dict[groupName] = group

            members = self.ADMIN.enumerateUsersByGroup(groupName)

            if not members:
                members = []
                
            members.sort()

            self.member_dict[groupName] = members

    def refresh_users_and_groups(self, names):
        self.busy()
        try:
            self._refresh_users_and_groups(names)
            self.ready()
        except:
            self.ready()
            messageDialog.show_ldap_error_dialog()
        
    def _refresh_users_and_groups(self, names):
        updated_names = {}
        for name in names:
            # If we've already updated this user/group, skip it.
            if updated_names.has_key(name):
                continue
            updated_names[name] = name
            service_pending_events()

            userEnt = self.ADMIN.lookupUserByName(name)
            if userEnt:
                self.user_dict[name] = userEnt
                gidNumber = userEnt.get(libuser.GIDNUMBER)[0]
                try:
                    groupEnt = self.gid_dict[long (gidNumber)]
                except:
                    try:
                        groupEnt = self.ADMIN.lookupGroupById(long (gidNumber))
                        self.gid_dict[long (gidNumber)] = groupEnt
                    except:
                        pass
            else:
                try:
                    del(self.user_dict[name])
                except:
                    pass

            groupEnt = self.ADMIN.lookupGroupByName(name)
            if groupEnt:
                self.group_dict[name] = groupEnt
                self.member_dict[name] = self.ADMIN.enumerateUsersByGroup(name)
                gidNumber = groupEnt.get(libuser.GIDNUMBER)
                gidNumber = gidNumber[0]
                self.gid_dict[long (gidNumber)] = groupEnt
            else:
                try:
                    del(self.group_dict[name])
                    del(self.member_dict[name])
                except:
                    pass

        users = self.get_user_list(self.get_filter_data())
        for user in self.user_dict.keys():
            if user not in users:
                del self.user_dict[user]

        groups = self.get_group_list(self.get_filter_data())
        for group in self.group_dict.keys():
            if group not in groups:
                del self.group_dict[group]

        self.refresh()

    def refresh(self, *args):
        self.busy()
        try:
            self.refresh_users()
            self.refresh_groups()
            self.populate_lists()
            self.toggleWidgets(gtk.FALSE)
            self.ready()
        except:
            self.ready()
            messageDialog.show_ldap_error_dialog()

    def populate_user_list(self):
        self.userStore.clear()
        for user in self.user_dict.keys():
            userEnt = self.user_dict[user]
            uid = userEnt.get(libuser.USERNAME)[0]
            uidNumber = userEnt.get(libuser.UIDNUMBER)[0]
            gidNumber = userEnt.get(libuser.GIDNUMBER)[0]
            if userEnt.get(libuser.GECOS):
                gecos = userEnt.get(libuser.GECOS)[0]
            else:
                gecos = uid
#            if userEnt.get(libuser.HOMEDIRECTORY):
#                homeDir = userEnt.get(libuser.HOMEDIRECTORY)[0]
#            else:
#                homeDir = ''
#            if userEnt.get(libuser.LOGINSHELL):
#                shell = userEnt.get(libuser.LOGINSHELL)[0]
#            else:
#                shell = ''

            try:
                groupEnt = self.gid_dict[long (gidNumber)]
                groupName = groupEnt.get(libuser.GROUPNAME)[0]
            except:
                groupName = gidNumber
                
            emailAddress = userEnt.get(libuser.USEREMAIL)[0]
            groupMemberships = self.ADMIN.enumerateGroupsByUser(uid)
            groupMemberships.sort()
            groupMembershipsStr = string.join(groupMemberships, "; ")

            #Convert the user fullName into unicode so pygtk2 doesn't truncate strings
            gecos = unicode (gecos, 'utf-8')

            iter = self.userStore.append()
            self.userStore.set_value(iter, 0, uid)
            self.userStore.set_value(iter, 1, uidNumber)
            self.userStore.set_value(iter, 2, groupName)
            self.userStore.set_value(iter, 3, gecos)
            self.userStore.set_value(iter, 4, emailAddress)
            self.userStore.set_value(iter, 5, groupMembershipsStr)
            self.userStore.set_value(iter, 6, userEnt)

    def populate_group_list(self):
        self.groupStore.clear()
        members = []

        for group in self.group_dict.keys():
            groupEnt = self.group_dict[group]

            if groupEnt.get(libuser.GIDNUMBER) != []:
                cn = groupEnt.get(libuser.GROUPNAME)[0]
                gid = groupEnt.get(libuser.GIDNUMBER)[0]
                members = self.member_dict[group]
                if not members:
                    members = []

            #concatenate the list of members in the group into a comma separated list
            memberlist = string.join(members, ", ")

            iter = self.groupStore.append()
            self.groupStore.set_value(iter, 0, cn)
            self.groupStore.set_value(iter, 1, gid)
            self.groupStore.set_value(iter, 2, memberlist)
            self.groupStore.set_value(iter, 3, groupEnt)
                
    def populate_lists(self):
        self.populate_user_list()
        self.populate_group_list()
        pass

    #------------Event handlers-------------#
    #---------------------------------------#

    #--------Event handlers for toplevel window-----#
    def on_about_button_clicked(self, *args):
        dlg = gnome.ui.GnomeAbout("Zarafa User Manager",
                                  "5.10",
                                  "Copyright (c) 2001 Red Hat, Inc. , 2007 Zarafa",
                                  ["Brent Fox <bfox@redhat.com>", "Ryan Hughes <ryan.hughes@zarafa.com"],
                                  _("This software is distributed under the GPL. "))
        dlg.run_and_close()

    def on_help_button_clicked(self, *args):
        help_pages = ["file:///usr/share/doc/zarafa-config-users-1.2.27/user-new.html",
                      "file:///usr/share/doc/zarafa-config-users-1.2.27/group-new.html",]
        page = help_pages [self.notebook.get_current_page ()]
        path = "/usr/bin/htmlview"
        path = None

        if path == None:
            messageDialog.show_message_dialog(_("Help is not available."))
            return
        
        pid = os.fork()
        if not pid:
            os.execv(path, [path, page])
        
    def on_manual_button_clicked(self, *args):
        page = "file:///usr/share/doc/zarafa-config-users-1.2.27/index.html"
        path = "/usr/bin/htmlview"
        path = None

        if path == None:
            messageDialog.show_message_dialog(_("Help is not available."))
            return
        
        pid = os.fork()
        if not pid:
            os.execv(path, [path, page])

    def on_exit1_activate(self, *args):
        if gtk.__dict__.has_key ("main_quit"):
            gtk.main_quit()
        else:
            gtk.mainquit()

    def on_add_user_activate(self, *args):
        try:
            filter = self.get_filter_data()
            self.userWin.newUserWin(filter)
            self.userWin.userWin.set_transient_for(self.toplevel)
        except:
            messageDialog.show_ldap_error_dialog()

    def on_add_group_activate(self, *args):
        try:
            self.groupProperties.newGroupWin()
            self.groupProperties.groupWin.set_transient_for(self.toplevel)
        except:
            messageDialog.show_ldap_error_dialog()

    def on_delete_activate(self, *args):
        page = self.notebook.get_current_page()
        if page == 0:
            if self.userTreeView.get_selection().get_selected():
                data, iter = self.userTreeView.get_selection().get_selected()
                userEnt = self.userStore.get_value(iter, 6)

                user = userEnt.get(libuser.USERNAME)[0]
                uid = userEnt.get(libuser.UIDNUMBER)[0]

                errMsgs = []

                ts = rpm.TransactionSet ()
                
                text = _("The user '%s' is about to be removed.  Are you sure you want to do this?" % user)
                
                dlg = gtk.MessageDialog(None, 0, gtk.MESSAGE_QUESTION, gtk.BUTTONS_YES_NO, text)
                dlg.set_position(gtk.WIN_POS_CENTER)
                dlg.set_modal(gtk.TRUE)
                dlg.set_icon(iconPixbuf)
                dlg.show_all()
                                                                                
                rc = dlg.run()

                if rc == gtk.RESPONSE_NO:
                    #bail out
                    dlg.destroy()
                    return
                else:
                    #Ok, we're going to delete the user
                    self.ADMIN.deleteUser(userEnt)
                    dlg.destroy()
                                                                                
                try:
                    del self.user_dict[user]
                except:
                    pass

                need_refresh = [user]
                for group in self.group_dict.keys():
                    groupEnt = self.group_dict[group]
                    members = groupEnt.get(libuser.MEMBERNAME)

                    if members:
                        if user in members:
                            members.remove(user)
                            groupEnt.set(libuser.MEMBERNAME, members)
                            self.ADMIN.modifyGroup(groupEnt)

                    self.userSelectedRow = None

                self.refresh_users_and_groups(need_refresh)

        elif page == 1:
            if self.groupTreeView.get_selection().get_selected():
                data, iter = self.groupTreeView.get_selection().get_selected()
                groupEnt = self.groupStore.get_value(iter, 3)
                groupName = groupEnt.get(libuser.GROUPNAME)[0]

                rc = messageDialog.show_confirm_dialog(_("Are you sure you want to delete the group '%s'?"
                                                         % groupName))
                if rc == gtk.RESPONSE_NO:
                    return
                        
                self.ADMIN.deleteGroup(groupEnt)
                try:
                    del self.group_dict[groupName]
                except:
                    pass
                self.refresh_users_and_groups(groupName)

    def on_properties_activate(self, *args):        
        page = self.notebook.get_current_page()
        if page == 0:
            self.user_properties()
        elif page == 1:
            self.group_properties()

    def user_properties(self):
        if self.userTreeView.get_selection().get_selected():
            data, iter = self.userTreeView.get_selection().get_selected()
        
            userEnt = self.userStore.get_value(iter, 6)
            self.userProperties.showUserProperties(userEnt)
            self.userProperties.userWin.set_transient_for(self.toplevel)

    def group_properties(self):
        if self.groupTreeView.get_selection().get_selected():
            data, iter = self.groupTreeView.get_selection().get_selected()
            groupEnt = self.groupStore.get_value(iter, 3)
            self.groupProperties.showGroupProperties(groupEnt)
            self.groupProperties.groupWin.set_transient_for(self.toplevel)

    def itemSelected(self, *args):
        # When an item is selected, sensitize the properties and the delete buttons.
        # When an item is unselected, desensitize the properties and delete buttons.
        page = self.notebook.get_current_page()

        if page == 0:
            object, data = self.userTreeView.get_selection().get_selected()
        elif page == 1:
            object, data = self.groupTreeView.get_selection().get_selected()

        if data == None:
            self.toggleWidgets(gtk.FALSE)
        else:
            self.toggleWidgets(gtk.TRUE)

    def changeNotebookPage(self, *args):
        # When the user changes a notebook page, unselect all the choices on both the user and group lists
        # This keeps the program from getting confused on which widget is currently selected
        page = self.notebook.get_current_page()

        if page == 0:
            self.groupTreeView.get_selection().unselect_all()
        elif page == 1:
            self.userTreeView.get_selection().unselect_all()

        self.toggleWidgets(gtk.FALSE)
        
    def toggleWidgets(self, value):
        self.properties_button.set_sensitive(value)
        self.properties_menu.set_sensitive(value)
        self.delete_button.set_sensitive(value)
        self.delete_menu.set_sensitive(value)
        
    def get_filter_data(self):
        filter = self.filter.get_text()
        filter = string.strip(filter)

        if len(filter) == 0:
            filter = ''
        else:
            index = string.find(filter, '*')
            length = len(filter) - 1

            if index != length:
                filter = filter + "*"
        return filter

    def rmrf(self, path):
        if path != "/dev/null":  #Don't allow /dev/null to be deleted
            # just use rm to avoid having to worry about races
            args = [ "/bin/rm", "-rf", "%s" %(path,) ]
            rhpl.executil.execWithRedirect(args[0], args)

    def do_rm_userowned (self, path, uid):
        if os.path.isdir (path) and not os.path.islink (path):
            for file in os.listdir (path):
                self.do_rm_userowned (path + os.sep + file, uid)
            if os.stat (path).st_uid == uid:
                try:
                    os.rmdir (path)
                except OSError:
                    pass
        else:
            if os.stat (path).st_uid == uid:
                try:
                    os.unlink (path)
                except OSError:
                    pass

    def rmtmpfiles (self, tmppaths, uid):
        for path in tmppaths:
            self.do_rm_userowned (path, uid)
        
    def rowActivated(self, *args):
        self.on_properties_activate()

    def isSELinuxInstalled(self):
        ts = rpm.TransactionSet()

        mi = ts.dbMatch('name', 'policy-sources')
        if mi.count() > 0:
            return 1
        return 0

    def isSELinuxEnabled(self):
        if self.isSELinuxInstalled():
            if os.system("/usr/bin/selinuxenabled") > 0:
                #it's enabled, return 1
                return 1
            else:
                #it's installed, but not enabled
                return 0
        else:
            #not installed
            return 0
